#pragma once

#include <ymod_httpclient/call.h>
#include <ymod_tvm/tvm.h>
#include <ymod_tvm/settings.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>
#include <yplatform/spinlock.h>

#include <boost/asio/deadline_timer.hpp>
#include <map>
#include <memory>

namespace ymod_tvm { namespace tvm2 {

class impl
    : public interface
    , public yplatform::module
{
    using lock_t = yplatform::spinlock;
    using guard_t = std::lock_guard<lock_t>;
    using tvm2_callback = tvm2_module::tickets_ready_callback;

public:
    impl(yplatform::reactor& reactor, const yplatform::ptree& config);
    impl(yplatform::reactor& reactor, settings settings);
    void start();
    void stop();

    // tvm2_module.
    void subscribe_service_ticket(const string& service, const tvm2_callback& cbs) override;

    void subscribe_all_tickets_are_ready(const callback& cbs) override;

    error_code check_service_ticket(task_context_ptr ctx, const string& ticket) override;

    boost::optional<service_ticket> get_native_service_ticket(
        task_context_ptr ctx,
        const string& ticket) override;

    boost::variant<error_code, service_ticket> get_native_service_ticket_or_error(
        const string& ticket) override;

    error_code get_service_ticket(const std::string& service, std::string& ticket) override;

    std::tuple<error_code, std::string> get_service_ticket_for_host(
        const std::string& host) override;

    virtual void bind_host(const std::string& host, const std::string& service) override;

    error_code check_user_ticket(task_context_ptr ctx, blackbox_env env, const string& ticket)
        override;

    boost::optional<user_ticket> get_native_user_ticket(
        task_context_ptr ctx,
        blackbox_env env,
        const string& ticket) override;

    boost::variant<error_code, user_ticket> get_native_user_ticket_or_error(
        blackbox_env env,
        const string& ticket) override;

    void subscribe_keys_loaded(const callback& cb) override;

private:
    void schedule_update_keys(duration interval);
    void update_keys(const error_code& error = error_code());
    void update_keys_from_file();
    void update_keys_from_api();
    void schedule_update_tickets(duration interval);
    void update_tickets(const error_code& error = error_code());
    bool load_keys(const string& body);
    bool parse_tickets(const string& resp_body);
    void check_not_started();
    void invoke_ticket_callbacks();
    void invoke_keys_callbacks();

    std::shared_ptr<std::map<string, string>> tickets_by_service_name()
    {
        guard_t guard(lock_);
        return state_.tickets_by_service_name_;
    }

    void tickets_by_service_name(const std::shared_ptr<std::map<string, string>>& new_tickets)
    {
        guard_t guard(lock_);
        state_.tickets_by_service_name_ = new_tickets;
    }

    void user_contexts(const std::shared_ptr<std::map<blackbox_env, user_context>>& new_contexts)
    {
        guard_t guard(lock_);
        state_.user_contexts_ = new_contexts;
    }

    std::shared_ptr<std::map<blackbox_env, user_context>> user_contexts()
    {
        guard_t guard(lock_);
        return state_.user_contexts_;
    }

    std::shared_ptr<service_context> service_check_context()
    {
        guard_t guard(lock_);
        return state_.service_check_context_;
    }

    void service_check_context(const std::shared_ptr<service_context>& new_context)
    {
        guard_t guard(lock_);
        state_.service_check_context_ = new_context;
    }

    auto state()
    {
        guard_t guard(lock_);
        return state_;
    }

    lock_t lock_;
    settings settings_;
    yplatform::reactor& reactor_;
    std::shared_ptr<timer> keys_timer_;
    std::shared_ptr<timer> tickets_timer_;
    std::shared_ptr<yhttp::cluster_client> http_client_;
    std::map<string, std::vector<tvm2_callback>> callbacks_;
    std::vector<callback> all_tickets_ready_callbacks_;
    std::vector<callback> keys_ready_callbacks_;
    string libtvm_version_;
    std::shared_ptr<service_context> service_sign_context_;
    bool started_ = false;
    task_context_ptr ctx_;

    struct
    {
        std::shared_ptr<service_context> service_check_context_;
        std::shared_ptr<std::map<blackbox_env, user_context>> user_contexts_;
        std::shared_ptr<std::map<string, string>> tickets_by_service_name_;
    } state_;
};

}}
