#include "blackbox_env.h"
#include "read_file.h"
#include <ymod_tvm/settings.h>
#include <boost/algorithm/string/trim.hpp>
#include <boost/lexical_cast.hpp>
#include <limits>

namespace ymod_tvm { namespace tvm2 {
namespace {

uint32_t get_id(const string& name, const yplatform::ptree& conf)
{
    if (auto id_node = conf.get_child_optional("id"))
    {
        return id_node->get_value<uint32_t>();
    }
    else if (auto env_id = blackbox_tvm_id_by_name(name))
    {
        return boost::lexical_cast<uint32_t>(env_id);
    }
    throw std::runtime_error(
        "no id provided for service " + name + " and it's not a known blackbox environment");
}
}

settings::mapping_t settings::read_mappings(const yplatform::ptree& conf, const string& node)
{
    std::map<string, service_info> mappings;
    if (auto mappings_node = conf.get_child_optional(node))
    {
        for (auto& [name, value] : *mappings_node)
        {
            if (mappings.count(name))
            {
                throw std::runtime_error("mapping for " + name + " already exists");
            }
            service_info info{ get_id(name, value), {} };
            yplatform::read_ptree(info.hosts, value, "hosts");
            yplatform::read_ptree(info.hosts, value, "host");
            mappings.emplace(name, std::move(info));
        }
    }
    return mappings;
}

settings::settings()
{
    http.enable_logging = true;
    // Set default retry policy.
    http.retry_policy.max_attempts = 10;
    http.default_request_timeout *= http.retry_policy.max_attempts;
    http.retry_policy.backoff =
        ymod_httpclient::backoff_settings{ .base = yplatform::time_traits::milliseconds(100),
                                           .max = yplatform::time_traits::seconds(10) };
}

settings::settings(const yplatform::ptree& conf) : settings()
{
    auto mappings = read_mappings(conf);

    if (auto alias = conf.get("my_tvm_id", string{}); mappings.count(alias))
    {
        my_tvm_id = mappings[alias].id;
    }
    else
    {
        my_tvm_id = conf.get<uint32_t>("service_id", my_tvm_id);
        my_tvm_id = conf.get<uint32_t>("my_tvm_id", my_tvm_id);
    }

    keys_update_interval = conf.get<duration>("keys_update_interval");
    tickets_update_interval = conf.get<duration>("tickets_update_interval");
    retry_interval = conf.get<duration>("retry_interval");
    if (conf.get<bool>("fetch_v1_ticket", false))
    {
        throw std::runtime_error(
            "`true` specified for `fetch_v1_ticket`, but TVM 1.0 is no longer supported.");
    }
    log_debug_info = conf.get<bool>("log_debug_info", false);

    keys_request = conf.get("keys_query", "/2/keys");
    keys_file = conf.get("keys_file", "");
    tickets_request = conf.get("tickets_query", "/2/ticket/");

    std::set<string> service_names;
    auto destinations_range = conf.equal_range("target_services");
    if (destinations_range.first == destinations_range.second)
    {
        destinations_range = conf.equal_range("destinations");
    }
    for (auto it = destinations_range.first; it != destinations_range.second; ++it)
    {
        auto alias = it->second.get_value(string{});
        bool use_alias = mappings.count(alias);
        auto name = use_alias ? alias : it->second.get<string>("name");
        std::set<string> conf_hosts;
        yplatform::read_ptree(conf_hosts, it->second, "host");
        yplatform::read_ptree(conf_hosts, it->second, "hosts");
        auto& hosts = use_alias ? mappings[alias].hosts : conf_hosts;
        string id;
        if (use_alias)
        {
            id = std::to_string(mappings[alias].id);
        }
        else
        {
            id = std::to_string(get_id(name, it->second));
        }
        if (!service_names.insert(name).second)
        {
            throw std::runtime_error("duplicate service name " + name);
        }
        if (!target_services_by_id.insert(std::make_pair(id, name)).second)
        {
            throw std::runtime_error("duplicate service id " + id);
        }
        for (auto& host : hosts)
        {
            if (host.size() && !tvm_service_by_host.insert(std::make_pair(host, name)).second)
            {
                throw std::runtime_error("host " + host + " was already configured");
            }
        }
        if (target_services.size()) target_services += ',';
        target_services += id;
    }

    auto blackbox_envs_range = conf.equal_range("blackbox_environments");
    for (auto it = blackbox_envs_range.first; it != blackbox_envs_range.second; ++it)
    {
        auto name = it->second.get_value<string>();
        blackbox_env env;
        if (!blackbox_env_by_name(name, env))
        {
            throw std::runtime_error("unknown blackbox environment " + name);
        }
        blackbox_envs.insert(env);
    }

    tvm_secret = conf.get("tvm_secret", tvm_secret);
    tvm_secret = conf.get("secret", tvm_secret);

    if (tvm_secret.empty())
    {
        auto tvm_secret_file = conf.get("tvm_secret_file", string());
        tvm_secret_file = conf.get("secret_file", tvm_secret_file);
        if (tvm_secret_file.size())
        {
            tvm_secret = read_file(tvm_secret_file);
        }
    }

    boost::algorithm::trim(tvm_secret);

    wait_first_update_on_start = conf.get("wait_first_update_on_start", false);
    if (auto http_conf = conf.get_child_optional("http"))
    {
        http.parse_ptree(*http_conf);
    }
    if (http.nodes.empty())
    {
        std::string schema = conf.get<bool>("https", true) ? "https://" : "http://";
        http.nodes.push_back(schema + conf.get<string>("tvm_host"));
    }
}

}}
