#include <ymod_tvm/tvm.h>
#include <yplatform/util/shared_ptr_cast.h>
#include "impl.h"

namespace ymod_tvm {

tvm::tvm(tvm2::settings settings)
{
    reactor_ = std::make_shared<yplatform::reactor>();
    reactor_->init(1, 1);
    impl_ = std::make_shared<tvm2::impl>(*reactor_, std::move(settings));
    reactor_->run();
}

tvm::tvm(boost::asio::io_service& io, tvm2::settings settings)
{
    reactor_ = std::make_shared<yplatform::reactor>(std::make_shared<yplatform::io_pool>(io, 1));
    impl_ = std::make_shared<tvm2::impl>(*reactor_, std::move(settings));
}

tvm::tvm(yplatform::reactor& reactor, tvm2::settings settings)
{
    reactor_ =
        shared_ptr_cast<std::shared_ptr>::from(yplatform::reactor::make_not_owning_copy(reactor));
    impl_ = std::make_shared<tvm2::impl>(*reactor_, std::move(settings));
}

void tvm::start()
{
    impl_->start();
}

tvm::~tvm()
{
    if (impl_)
    {
        impl_->stop();
    }
    if (reactor_)
    {
        reactor_->stop();
        reactor_->fini();
    }
}

void tvm::subscribe_service_ticket(const std::string& service, const tickets_ready_callback& cbs)
{
    impl_->subscribe_service_ticket(service, cbs);
}

void tvm::subscribe_all_tickets_are_ready(const callback& cbs)
{
    impl_->subscribe_all_tickets_are_ready(cbs);
}

boost::system::error_code tvm::check_service_ticket(task_context_ptr ctx, const std::string& ticket)
{
    return impl_->check_service_ticket(ctx, ticket);
}

boost::optional<tvm2::service_ticket> tvm::get_native_service_ticket(
    task_context_ptr ctx,
    const std::string& ticket)
{
    return impl_->get_native_service_ticket(ctx, ticket);
}

boost::variant<error_code, tvm2::service_ticket> tvm::get_native_service_ticket_or_error(
    const std::string& ticket)
{
    return impl_->get_native_service_ticket_or_error(ticket);
}

boost::system::error_code tvm::get_service_ticket(const std::string& service, std::string& ticket)
{
    return impl_->get_service_ticket(service, ticket);
}

std::tuple<boost::system::error_code, std::string> tvm::get_service_ticket_for_host(
    const std::string& host)
{
    return impl_->get_service_ticket_for_host(host);
}

void tvm::bind_host(const std::string& host, const std::string& service)
{
    impl_->bind_host(host, service);
}

boost::system::error_code tvm::check_user_ticket(
    task_context_ptr ctx,
    tvm2::blackbox_env env,
    const std::string& ticket)
{
    return impl_->check_user_ticket(ctx, env, ticket);
}

boost::optional<tvm2::user_ticket> tvm::get_native_user_ticket(
    task_context_ptr ctx,
    tvm2::blackbox_env env,
    const std::string& ticket)
{
    return impl_->get_native_user_ticket(ctx, env, ticket);
}

boost::variant<error_code, tvm2::user_ticket> tvm::get_native_user_ticket_or_error(
    tvm2::blackbox_env env,
    const std::string& ticket)
{
    return impl_->get_native_user_ticket_or_error(env, ticket);
}

void tvm::subscribe_keys_loaded(const callback& cb)
{
    impl_->subscribe_keys_loaded(cb);
}

}
