#include "histogram_metric.h"

#include <algorithm>

namespace NYmodUnistat {

THistogramMetric::THistogramMetric(TBorders borders) {
    std::sort(borders.begin(), borders.end());
    Values.reserve(borders.size());

    for (auto border : borders) {
        Values.emplace_back(border, std::make_unique<std::atomic_uint64_t>(0));
    }
}

void THistogramMetric::Push(double value) {
    auto it = std::upper_bound(
        Values.begin(),
        Values.end(),
        TIntervalData{value, nullptr},
        [](const TIntervalData& a, const TIntervalData& b) {
            return a.first < b.first;
        });
    if (it == Values.begin()) {
        return;
    }
    it = std::prev(it);
    auto& count = *it->second;

    count.fetch_add(1, std::memory_order_acq_rel);
}

THistogramMetric::TValue THistogramMetric::GetValue(bool reset) {
    THistogramValues values;
    values.reserve(Values.size());

    for (auto& [border, countPtr] : Values) {
        const auto count = reset ?
            countPtr->exchange(0, std::memory_order_acq_rel) :
            countPtr->load(std::memory_order_acquire);

        values.emplace_back(border, count);
    }

    return values;
}

void THistogramMetric::Reset() {
    for (auto& [border, countPtr] : Values) {
        countPtr->store(0, std::memory_order_release);
    }
}

} // namespace NYmodUnistat
