#pragma once

#include "unistat.h"

#include "metrics/histogram_metric.h"
#include "metrics/average_metric.h"
#include "metrics/last_value_metric.h"
#include "metrics/max_metric.h"
#include "metrics/min_metric.h"
#include "metrics/sum_metric.h"
#include "metrics/sum_none_metric.h"
#include "metrics/metric.h"

#include <yplatform/ptree.h>

#include <library/cpp/json/json_writer.h>
#include <util/stream/str.h>
#include <util/system/rwlock.h>

#include <list>
#include <memory>
#include <string>
#include <unordered_map>

namespace NYmodUnistat {

class TModuleImpl {
public:
    void Init(const yplatform::ptree& configuration);

    void Push(const std::string& signal, double value);

    TMetric::TValue GetValue(const std::string& signal, bool reset);
    IUnistat::TValues GetValues(bool reset);
    std::string GetValuesInJson(bool reset);

    void ResetAll();

    TMetricPtr SetHandler(const std::string& signal, TMetricPtr metric);

    bool IsMetricPresent(const std::string& signal);
    void AddMetric(const yplatform::ptree& config);

private:
    void ReadMetricsConfiguration(const yplatform::ptree& configuration);

    TMetricPtr CreateMetric(const yplatform::ptree& configuration);
    TMetricPtr CreateHistogramMetric(THistogramMetric::TBorders borders);
    TMetricPtr CreateNumericalMetric(const std::string& aggregation, double startValue);

    std::string AggregationTypeToSuffix(const std::string& str);

private:
    struct TMetricDesc {
        std::string Signal;
        std::string Suffix;
        TMetricPtr Metric;
    };

    using TMetricDescs = std::list<TMetricDesc>;

    TMetricDescs MetricsList;
    std::unordered_map<std::string, TMetricDescs::iterator> MetricsDict;
    // TODO: make Lock mutable
    // and place const in declarations of appropriate methods
    TRWMutex Lock;
};

void TModuleImpl::Init(const yplatform::ptree& configuration) {
    ReadMetricsConfiguration(configuration);
}

bool TModuleImpl::IsMetricPresent(const std::string& signal) {
    TReadGuard guard(Lock);
    const auto it = MetricsDict.find(signal);
    return it != MetricsDict.end();
}

void TModuleImpl::AddMetric(const yplatform::ptree& config) {
    auto signal = config.get<std::string>("name", "");
    if (signal.empty()) {
        throw std::runtime_error("Name of unistat metric must not be empty");
    }

    auto aggregation = config.get<std::string>("aggregation", "");
    if (aggregation.empty()) {
        throw std::runtime_error("Invalid aggregation method");
    }

    std::string suffix = aggregation[0] == '_' ? aggregation : AggregationTypeToSuffix(aggregation);
    if (suffix.empty()) {
        throw std::runtime_error("Invalid aggregation method");
    }

    TWriteGuard guard(Lock);
    const auto it = MetricsDict.find(signal);
    if (it != MetricsDict.end()) {
        // metric with specified name is already present
        return;
    }

    MetricsList.push_back({signal, std::move(suffix), CreateMetric(config)});
    MetricsDict.emplace(std::move(signal), std::prev(MetricsList.end()));
}

void TModuleImpl::ReadMetricsConfiguration(const yplatform::ptree& configuration) {
    for (const auto& [name, pt] : configuration) {
        if (name != "metrics") {
            continue;
        }

        AddMetric(pt);
    }
}

TMetricPtr TModuleImpl::CreateMetric(const yplatform::ptree& configuration) {
    TMetricPtr metric;
    auto hostAggregation = configuration.get<std::string>("host_aggregation");

    if (hostAggregation == "histogram") {
        THistogramMetric::TBorders borders;
        auto bordersRange = configuration.equal_range("borders");
        for (auto border = bordersRange.first; border != bordersRange.second; ++border) {
            borders.push_back(border->second.get_value<double>());
        }
        if (borders.empty()) {
            throw std::runtime_error("Borders of histogram interval must be defined");
        }
        metric = CreateHistogramMetric(std::move(borders));
    } else if (hostAggregation != "empty") {
        auto startValue = configuration.get<double>("start_value", 0);
        metric = CreateNumericalMetric(hostAggregation, startValue);
        if (!metric) {
            throw std::runtime_error("Invalid host aggregation method");
        }
    }

    return metric;
}

void TModuleImpl::Push(const std::string &signal, double value) {
    TReadGuard guard(Lock);
    const auto it = MetricsDict.find(signal);
    if (it == MetricsDict.end()) {
        throw std::invalid_argument("Fail to find metric " + signal);
    }
    auto& metric = it->second->Metric;
    if (!metric) {
        throw std::runtime_error("Metric is not initialized");
    }
    metric->Push(value);
}

TMetric::TValue TModuleImpl::GetValue(const std::string& signal, bool reset) {
    TReadGuard guard(Lock);
    const auto it = MetricsDict.find(signal);
    if (it == MetricsDict.end()) {
        throw std::invalid_argument("Fail to find metric " + signal);
    }
    auto& metric = it->second->Metric;
    if (!metric) {
        throw std::runtime_error("Metric is not initialized");
    }
    return metric->GetValue(reset);
}

IUnistat::TValues TModuleImpl::GetValues(bool reset) {
    TReadGuard guard(Lock);
    IUnistat::TValues values;
    for (auto& desc : MetricsList) {
        values.push_back({desc.Signal, desc.Suffix, desc.Metric ? desc.Metric->GetValue(reset) : TMetric::TValue{}});
    }
    return values;
}

std::string TModuleImpl::GetValuesInJson(bool reset) {
    TStringStream stream;
    NJson::TJsonWriterConfig config;
    config.ValidateUtf8 = false;

    NJson::TJsonWriter writer(&stream, config);
    writer.OpenArray();

    TReadGuard guard(Lock);
    for (const auto& desc : MetricsList) {
        if (!desc.Metric) {
            continue;
        }
        const auto value = desc.Metric->GetValue(reset);
        if (std::holds_alternative<std::monostate>(value)) {
            continue;
        }

        writer.OpenArray();
        writer.Write(desc.Signal + desc.Suffix);

        if (auto numericVal = std::get_if<double>(&value)) {
            writer.Write(*numericVal);
        } else {
            auto histogramVal = std::get<TMetric::THistogramValues>(value);
            writer.OpenArray();
            for (auto& [border, count] : histogramVal) {
                writer.OpenArray();
                writer.Write(border);
                writer.Write(count);
                writer.CloseArray();
            }
            writer.CloseArray();
        }

        writer.CloseArray();
    }

    writer.CloseArray();
    writer.Flush();

    return static_cast<std::string>(stream.Str());
}

void TModuleImpl::ResetAll() {
    TReadGuard guard(Lock);
    for (auto& desc : MetricsList) {
        if (desc.Metric) {
            desc.Metric->Reset();
        }
    }
}

TMetricPtr TModuleImpl::SetHandler(const std::string& signal, TMetricPtr metric) {
    TWriteGuard guard(Lock);
    const auto it = MetricsDict.find(signal);
    if (it == MetricsDict.end()) {
        throw std::invalid_argument("Fail to find metric " + signal);
    }
    auto& originalMetric = it->second->Metric;
    std::swap(metric, originalMetric);
    return metric;
}

TMetricPtr TModuleImpl::CreateHistogramMetric(THistogramMetric::TBorders borders) {
    return std::make_unique<THistogramMetric>(std::move(borders));
}

TMetricPtr TModuleImpl::CreateNumericalMetric(const std::string& aggregation, double startValue) {
    if (aggregation == "average") {
        return std::make_unique<TAverageMetric>();
    }
    if (aggregation == "max") {
        return std::make_unique<TMaxMetric>(startValue);
    }
    if (aggregation == "min") {
        return std::make_unique<TMinMetric>(startValue);
    }
    if (aggregation == "sum") {
        return std::make_unique<TSumMetric>(startValue);
    }
    if (aggregation == "sumnone") {
        return std::make_unique<TSumNoneMetric>(startValue);
    }
    if (aggregation == "last_value") {
        return std::make_unique<TLastValueMetric>(startValue);
    }

    return nullptr;
}

std::string TModuleImpl::AggregationTypeToSuffix(const std::string& str) {
    static const std::unordered_map<std::string, std::string> dict({
        {"delta_max_sum", "_dxxm"},
        {"delta_histogram", "_dhhh"},
        {"delta_sum", "_deee"},
        {"absolute_histogram", "_ahhh"},
        {"absolute_max", "_axxx"},
        {"absolute_sum_max", "_ammx"},
        {"absolute_average", "_avvv"}});

    const auto it = dict.find(str);
    return it == dict.end() ? "" : it->second;
}

} // namespace NYmodUnistat
