#include <mail/ymod_unistat/lib/unistat.h>

#include <yplatform/find.h>

#include <boost/python.hpp>
#include <boost/property_tree/json_parser.hpp>

#include <memory>

namespace NPyUnistat {

namespace py = boost::python;

using TUnistatPtr = boost::shared_ptr<NYmodUnistat::IUnistat>;

using TPyUnistat = py::class_<NYmodUnistat::IUnistat, TUnistatPtr, boost::noncopyable>;

void PyErr(PyObject* exc, const std::string& msg) {
    PyErr_SetString(exc, msg.c_str());
    boost::python::throw_error_already_set();
}

NYmodUnistat::IUnistat* FindUnistat(const std::string& moduleName) {
    try {
        return yplatform::find<NYmodUnistat::IUnistat>(moduleName).get();
    } catch (const std::exception& e) {
        PyErr(PyExc_RuntimeError, e.what());
    }
    return nullptr;
}

bool IsMetricPresent(TUnistatPtr unistat, const std::string& signal) {
    if (!unistat) {
        PyErr(PyExc_TypeError, "Not an unistat module ptr");
        return false;
    } else {
        return unistat->IsMetricPresent(signal);
    }
}

void AddMetric(TUnistatPtr unistat, const std::string& json) {
    if (!unistat) {
        PyErr(PyExc_TypeError, "Not an unistat module ptr");
    } else {
        try {
            yplatform::ptree config;
            auto istream = std::basic_istringstream<yplatform::ptree::key_type::value_type>(json);
            boost::property_tree::json_parser::read_json<yplatform::ptree>(istream, config);
            unistat->AddMetric(config);
        } catch (const std::exception& e) {
            PyErr(PyExc_RuntimeError, e.what());
        }
    }
}

void Push(TUnistatPtr unistat, const std::string& signal, double value) {
    if (!unistat) {
        PyErr(PyExc_TypeError, "Not an unistat module ptr");
    } else {
        try {
            unistat->Push(signal, value);
        } catch (const std::exception& e) {
            PyErr(PyExc_RuntimeError, e.what());
        }
    }
}

std::string GetValues(TUnistatPtr unistat, bool reset = false) {
    if (!unistat) {
        PyErr(PyExc_TypeError, "Not an unistat module ptr");
    } else {
        try {
            return unistat->GetValuesInJson(reset);
        } catch (const std::exception& e) {
            PyErr(PyExc_RuntimeError, e.what());
        }
    }
    return {};
}

} // namespace NPyUnistat

namespace {

BOOST_PYTHON_FUNCTION_OVERLOADS(GetValuesOverloads, NPyUnistat::GetValues, 1, 2);

BOOST_PYTHON_MODULE(py_unistat) {
    using namespace NPyUnistat;
    using namespace py;

    auto c = TPyUnistat("TPyUnistat", no_init);
    c.def("is_metric_present", IsMetricPresent);
    c.def("add_metric", AddMetric);
    c.def("push", Push);
    c.def("get_values_in_json_str", GetValues, GetValuesOverloads(args("self", "reset")));

    // it is ok here to use return_value_policy<reference_existing_object>,
    // because lifetime of python interpreter is the same as 
    // ymod_unistat module
    def("find_module", FindUnistat, return_value_policy<reference_existing_object>());
}

} // namespace anonymous
