#include "custom_metric.h"

#include <mail/ymod_unistat/lib/module.h>

#include <yplatform/application/config/yaml_to_ptree.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/testing/unittest/gtest.h>

#include <memory>

using namespace NYmodUnistat;

const std::string moduleConfiguration = R"(
metrics:
-   name: metric1
    aggregation: delta_sum
    host_aggregation: sum
-   name: metric2
    aggregation: delta_max_sum
    host_aggregation: max
    start_value: 100
-   name: metric3
    aggregation: absolute_max
    host_aggregation: last_value
-   name: metric4
    aggregation: _annn
    host_aggregation: min
    start_value: 10000
-   name: metric5
    aggregation: absolute_average
    host_aggregation: average
-   name: metric6
    aggregation: absolute_sum_max
    host_aggregation: sumnone
-   name: metric7
    aggregation: absolute_histogram
    host_aggregation: histogram
    borders: [0.1, 0.5, 1.0, 3.0, 10.0]
-   name: metric8
    aggregation: delta_histogram
    host_aggregation: histogram
    borders: [50, 100, 10]
-   name: metric9
    aggregation: absolute_sum_max
    host_aggregation: empty
)";

class TTestUnistatModule : public ::NTesting::TTest {
public:
    void SetUp() override {
        yplatform::ptree configuration;
        utils::config::yaml_to_ptree::convert_str(moduleConfiguration, configuration);

        auto module = std::make_unique<TModule>();
        module->init(configuration);

        Module = std::move(module);
    }

    bool CompareHistValues(TMetric::THistogramValues actual, TMetric::THistogramValues expected) {
        return actual == expected;
    }

    void CheckEmpty(IUnistat::TValues values) {
        ASSERT_EQ(values.size(), 9);

        EXPECT_STREQ(values[0].Signal.c_str(), "metric1");
        EXPECT_STREQ(values[1].Signal.c_str(), "metric2");
        EXPECT_STREQ(values[2].Signal.c_str(), "metric3");
        EXPECT_STREQ(values[3].Signal.c_str(), "metric4");
        EXPECT_STREQ(values[4].Signal.c_str(), "metric5");
        EXPECT_STREQ(values[5].Signal.c_str(), "metric6");
        EXPECT_STREQ(values[6].Signal.c_str(), "metric7");
        EXPECT_STREQ(values[7].Signal.c_str(), "metric8");
        EXPECT_STREQ(values[8].Signal.c_str(), "metric9");

        EXPECT_STREQ(values[0].Suffix.c_str(), "_deee");
        EXPECT_STREQ(values[1].Suffix.c_str(), "_dxxm");
        EXPECT_STREQ(values[2].Suffix.c_str(), "_axxx");
        EXPECT_STREQ(values[3].Suffix.c_str(), "_annn");
        EXPECT_STREQ(values[4].Suffix.c_str(), "_avvv");
        EXPECT_STREQ(values[5].Suffix.c_str(), "_ammx");
        EXPECT_STREQ(values[6].Suffix.c_str(), "_ahhh");
        EXPECT_STREQ(values[7].Suffix.c_str(), "_dhhh");
        EXPECT_STREQ(values[8].Suffix.c_str(), "_ammx");

        auto numValue = std::get<double>(values[0].Value);
        EXPECT_DOUBLE_EQ(numValue, 0);
        numValue = std::get<double>(values[1].Value);
        EXPECT_DOUBLE_EQ(numValue, 100);
        numValue = std::get<double>(values[2].Value);
        EXPECT_DOUBLE_EQ(numValue, 0);
        numValue = std::get<double>(values[3].Value);
        EXPECT_DOUBLE_EQ(numValue, 10000);
        numValue = std::get<double>(values[4].Value);
        EXPECT_DOUBLE_EQ(numValue, 0);
        EXPECT_NE(std::get_if<std::monostate>(&values[5].Value), nullptr);
        auto histValue = std::get<TMetric::THistogramValues>(values[6].Value);
        EXPECT_TRUE(CompareHistValues(histValue, {{0.1, 0}, {0.5, 0}, {1.0, 0}, {3.0, 0}, {10.0, 0}}));
        histValue = std::get<TMetric::THistogramValues>(values[7].Value);
        EXPECT_TRUE(CompareHistValues(histValue, {{10, 0}, {50, 0}, {100, 0}}));
        EXPECT_NE(std::get_if<std::monostate>(&values[8].Value), nullptr);
    }

    TString RemakeJson(std::string str) {
        NJson::TJsonValue value;
        NJson::ReadJsonTree(str, &value, true);
        return NJson::WriteJson(value, false);
    }

protected:
    std::unique_ptr<IUnistat> Module;
};

TEST_F(TTestUnistatModule, GetEmpty) {
    CheckEmpty(Module->GetValues(true));
}

TEST_F(TTestUnistatModule, PushGetReset) {
    EXPECT_THROW(Module->Push("metric0", 1), std::invalid_argument);
    EXPECT_THROW(Module->GetValue("metric0", false), std::invalid_argument);

    Module->Push("metric1", 10);
    Module->Push("metric5", 20);
    Module->Push("metric5", 30);
    Module->Push("metric1", 40);

    auto numValue = std::get<double>(Module->GetValue("metric1", false));
    EXPECT_DOUBLE_EQ(numValue, 50);
    numValue = std::get<double>(Module->GetValue("metric5", false));
    EXPECT_DOUBLE_EQ(numValue, 25);

    numValue = std::get<double>(Module->GetValue("metric1", true));
    EXPECT_DOUBLE_EQ(numValue, 50);
    numValue = std::get<double>(Module->GetValue("metric5", true));
    EXPECT_DOUBLE_EQ(numValue, 25);

    CheckEmpty(Module->GetValues(true));

    Module->Push("metric1", 10);
    numValue = std::get<double>(Module->GetValue("metric1", true));
    EXPECT_DOUBLE_EQ(numValue, 10);
}

TEST_F(TTestUnistatModule, ResetAll) {
    Module->Push("metric2", 10);
    Module->Push("metric3", 20);
    Module->Push("metric4", 30);
    Module->Push("metric6", 40);

    Module->ResetAll();

    CheckEmpty(Module->GetValues(true));
}

TEST_F(TTestUnistatModule, CustomHandler) {
    EXPECT_THROW(Module->Push("metric9", 1), std::runtime_error);
    EXPECT_THROW(Module->GetValue("metric9", 1), std::runtime_error);

    int pushCallCount = 0;
    int valueCallCount = 0;
    int resetCallCount = 0;
    double currentValue = 0;

    auto metric = std::make_unique<NYmodUnistat::NTesting::TCustomMetric>(
        [&pushCallCount, &currentValue](double value) {
            ++pushCallCount;
            currentValue = value;
        },
        [&valueCallCount, &currentValue](bool reset) {
            ++valueCallCount;
            double res = currentValue;
            if (reset) {
                currentValue = 0;
            }
            return res;
        },
        [&resetCallCount, &currentValue]() {
            ++resetCallCount;
            currentValue = 0;
        }
    );

    Module->SetHandler("metric9", std::move(metric));

    double value = std::get<double>(Module->GetValue("metric9", false));
    EXPECT_DOUBLE_EQ(value, 0);

    Module->Push("metric9", 999);
    value = std::get<double>(Module->GetValue("metric9", true));
    EXPECT_DOUBLE_EQ(value, 999);
    value = std::get<double>(Module->GetValue("metric9", false));
    EXPECT_DOUBLE_EQ(value, 0);

    Module->Push("metric9", 888);
    value = std::get<double>(Module->GetValue("metric9", false));
    EXPECT_DOUBLE_EQ(value, 888);
    value = std::get<double>(Module->GetValue("metric9", false));
    EXPECT_DOUBLE_EQ(value, 888);

    Module->ResetAll();
    value = std::get<double>(Module->GetValue("metric9", false));
    EXPECT_DOUBLE_EQ(value, 0);

    EXPECT_EQ(pushCallCount, 2);
    EXPECT_EQ(valueCallCount, 6);
    EXPECT_EQ(resetCallCount, 1);
}

TEST_F(TTestUnistatModule, Json) {
    Module->Push("metric2", 20);
    Module->Push("metric2", 30);
    Module->Push("metric3", 40);
    Module->Push("metric3", 50);
    Module->Push("metric4", 60);
    Module->Push("metric4", 70);
    Module->Push("metric5", 80);
    Module->Push("metric5", 90);

    Module->Push("metric7", 2);
    Module->Push("metric7", 0.3);
    Module->Push("metric7", 10);
    Module->Push("metric7", 0);

    Module->Push("metric8", 11);
    Module->Push("metric8", 1000);
    Module->Push("metric8", 5);
    Module->Push("metric8", 99);

    const std::string expectedJson1 = R"([
        ["metric1_deee", 0.0],
        ["metric2_dxxm", 100.0],
        ["metric3_axxx", 50.0],
        ["metric4_annn", 60.0],
        ["metric5_avvv", 85.0],
        ["metric7_ahhh", [[0.1, 1], [0.5, 0], [1.0, 1], [3.0, 0], [10.0, 1]]],
        ["metric8_dhhh", [[10.0, 1], [50.0, 1], [100.0, 1]]]
    ])";

    EXPECT_STREQ(RemakeJson(expectedJson1), RemakeJson(Module->GetValuesInJson(false)));

    Module->GetValue("metric2", true);
    Module->GetValue("metric3", true);
    Module->Push("metric6", 1000);
    Module->Push("metric6", 2000);

    const std::string expectedJson2 = R"([
        ["metric1_deee", 0.0],
        ["metric2_dxxm", 100.0],
        ["metric3_axxx", 0.0],
        ["metric4_annn", 60.0],
        ["metric5_avvv", 85.0],
        ["metric6_ammx", 3000.0],
        ["metric7_ahhh", [[0.1, 1], [0.5, 0], [1.0, 1], [3.0, 0], [10.0, 1]]],
        ["metric8_dhhh", [[10.0, 1], [50.0, 1], [100.0, 1]]]
    ])";

    EXPECT_STREQ(RemakeJson(expectedJson2), RemakeJson(Module->GetValuesInJson(true)));

    CheckEmpty(Module->GetValues(true));
}
