#include <string>
#include <sstream>
#include <yplatform/module.h>
#include <yplatform/repository.h>
#include <yplatform/find.h>
#include <ymod_webserver/server.h>
#include <ymod_webserver/handler.h>

namespace ymod_webservertest {
using std::string;
using std::stringstream;
using ymod_webserver::request_ptr;
using ymod_webserver::response_ptr;
using ymod_webserver::http_error;
using ymod_webserver::websocket::output_stream;
using ymod_webserver::websocket::output_stream_ptr;
typedef ymod_webserver::websocket::message ws_message_t;

class websocket_handler : public boost::enable_shared_from_this<websocket_handler>
{
    typedef boost::function<void(void)> close_hook_t;

public:
    websocket_handler(output_stream_ptr stream) : output(stream), is_binary(false)
    {
    }

    ~websocket_handler()
    {
        YLOG_G(debug) << "destroyed websocket_handler " << static_cast<void*>(this);
    }

    void init()
    {
        output->add_message_callback(
            boost::bind(&websocket_handler::on_message, weak_from_this(), _1));
        output->set_close_callback(
            boost::bind(&websocket_handler::close, weak_from_this(), _1, _2));
        output->begin_receive();
    }

    static void on_message(boost::weak_ptr<websocket_handler> weak_handler, ws_message_t const& msg)
    {
        if (auto handler = weak_handler.lock())
        {
            switch (msg.opcode)
            {
            case ws_message_t::opcode_text:
            case ws_message_t::opcode_binary:
                handler->is_binary = msg.opcode == ws_message_t::opcode_binary;
            case ws_message_t::opcode_continuation:
                handler->os << string(msg.data.begin(), msg.data.end());
                if (msg.is_finished())
                {
                    handler->perform_message();
                    handler->os.str(string());
                }
                break;
            default:
                break;
            }
        }
        else
        {
            YLOG_G(error) << "error: handler was already destroyed when message received";
        }
    }

    void perform_message()
    {
        if (is_binary) output->send_binary(os.str());
        else
            output->send_text(os.str());
    }

    static void close(
        boost::weak_ptr<websocket_handler> weak_handler,
        uint16_t /*code*/,
        const std::string& reason)
    {
        if (auto handler = weak_handler.lock())
        {
            YLOG_G(debug) << "closing websocket_handler: " << reason;
            handler->output->close_connection();
            handler->output.reset();
            handler->close_hook();
        }
        else
        {
            YLOG_G(error) << "error: handler was already destroyed when message received";
        }
    }

    void set_close_callback(close_hook_t hook)
    {
        close_hook = hook;
    }

    boost::weak_ptr<websocket_handler> weak_from_this()
    {
        return boost::dynamic_pointer_cast<websocket_handler>(shared_from_this());
    }

private:
    output_stream_ptr output;
    stringstream os;
    bool is_binary;

    close_hook_t close_hook;
};

class handler : public ymod_webserver::handler
{
public:
    void execute(request_ptr req, response_ptr stream)
    {
        YLOG_G(debug) << "* execute called";
        if (req->url.path.size() && req->url.path[0] == "ping")
        {
            stream->result(ymod_webserver::codes::ok, "pong");
        }
        else if (req->url.path.size() && req->url.path[0] == "wstest")
        {
            // just wait
        }
        else
        {
            stream->result(ymod_webserver::codes::bad_request, "Bro, say 'ping'!");
        }
    }

    void upgrade_to_websocket(request_ptr req)
    {
        if (req->url.path.size() && req->url.path[0] == "wstest")
        {
            // for echo server
        }
        else
        {
            throw http_error(
                "handler", "Not Implemented", "websocket_upgrade method not implemented")
                << ymod_webserver::http_result_code(ymod_webserver::codes::not_implemented);
        }
    }

    void execute_websocket(ymod_webserver::websocket::output_stream_ptr stream)
    {
        auto ws_handler = boost::make_shared<websocket_handler>(stream);
        ws_handler->init();
        stream->set_close_callback(
            boost::bind(&handler::close_ws, this, ws_handler->weak_from_this()));
        active_handlers.insert(ws_handler);
    }

    void close_ws(boost::weak_ptr<websocket_handler> ws_handler)
    {
        active_handlers.erase(ws_handler.lock());
    }

private:
    std::set<boost::shared_ptr<websocket_handler>> active_handlers;
};

class module : public yplatform::module
{
public:
    void init(const yplatform::ptree& /*xml*/)
    {
        handler_ = boost::make_shared<handler>();
        auto webserver = yplatform::find<ymod_webserver::server>("web_server");
        webserver->subscribe("endpoint1", handler_);
    }

private:
    boost::shared_ptr<handler> handler_;
};

} // namespace ymod_webserver

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(ymod_webservertest::module)
