#pragma once

#include <yplatform/task_context.h>
#include <yplatform/util/mtimer.h>
#include <ymod_webserver/request_base.h>
#include <boost/atomic/atomic.hpp>
#include <utility>

namespace ymod_webserver {

struct request;

enum class context_state_t
{
    start,
    handshake,
    read_headers,
    read_body,
    execute
};

typedef uint32_t prof_time_t;
typedef yplatform::util::mtimer_t prof_timer_t;

struct prof_value_item
{

    string name;
    prof_time_t time;
};

inline string prof_time_to_string(prof_time_t value)
{
    std::string buf;
    buf.resize(14);
    int ret = snprintf(&buf[0], buf.size(), "%.3f", value / 1000.0);
    return (ret > 0 && ret < 14) ? (buf.resize(ret), buf) : string("inval");
}

class prof_accumulator
{
    struct prof_item
    {
        string name;
        prof_timer_t timer;
    };

public:
    prof_accumulator(string id) : id_(std::move(id))
    {
    }

    ~prof_accumulator()
    {
        try
        {
            for (auto& prof_timer : prof_timers_)
            {
                YLOG_G(error) << id_ << " ~prof_accumulator profile data leak '" << prof_timer.name
                              << "'";
            }
        }
        catch (...)
        {
        }
    }

    void push(const string& name)
    {
        try
        {
            boost::mutex::scoped_lock lock(mux_);
            prof_timers_.push_back(prof_item({ name, prof_timer_t() }));
        }
        catch (...)
        {
        }
    }

    void pop(const string& name, bool allow_nulls = true)
    {
        try
        {
            boost::mutex::scoped_lock lock(mux_);
            for (auto it = prof_timers_.begin(); it != prof_timers_.end(); ++it)
            {
                if (it->name == name)
                {
                    auto time = it->timer.stop();
                    if (time != 0 || allow_nulls)
                    {
                        prof_values_.push_back(prof_value_item{ name, time });
                    }
                    prof_timers_.erase(it);
                    return;
                }
            }
            YLOG_G(error) << id_ << " prof_accumulator::pop element not found '" << name << "'";
        }
        catch (...)
        {
        }
    }

    // finish the last interval and start a new one
    void extrude_and_push(const string& name, bool allow_nulls = true)
    {
        pop(allow_nulls);
        push(name);
    }

    // pop the last interval
    void pop(bool allow_nulls = true)
    {
        boost::mutex::scoped_lock lock(mux_);
        try
        {
            if (prof_timers_.empty()) return;
            auto& val = prof_timers_.back();
            auto time = val.timer.stop();
            if (time != 0 || allow_nulls)
            {
                prof_values_.push_back(prof_value_item{ val.name, time });
            }
            prof_timers_.pop_back();
            return;
        }
        catch (...)
        {
        }
    }

    string get_values_with_total()
    {
        try
        {
            std::vector<prof_value_item> copy = get_values();
            std::stringstream ss;
            for (auto& it : copy)
            {
                ss << it.name << "=" << prof_time_to_string(it.time) << " ";
            }
            ss << "duration=" << prof_time_to_string(get_total());
            return ss.str();
        }
        catch (...)
        {
            return "";
        }
    }

    std::vector<prof_value_item> get_values() const
    {
        boost::mutex::scoped_lock lock(mux_);
        return prof_values_;
    }

    prof_time_t get_total() const
    {
        try
        {
            boost::mutex::scoped_lock lock(mux_);
            return total_prof_timer_.shot();
        }
        catch (...)
        {
            return 0;
        }
    }

private:
    string id_;

    std::vector<prof_value_item> prof_values_;
    std::vector<prof_item> prof_timers_;

    prof_timer_t total_prof_timer_;
    mutable boost::mutex mux_;
};

class context : public yplatform::task_context
{
public:
    context()
        : start_time(std::time(nullptr)), profilers(uniq_id()), bytes_sent(0), bytes_received(0)
    {
    }

    const string& get_name() const override
    {
        static const string name = "web_session";
        return name;
    }

    string local_address;
    unsigned local_port;

    string remote_address;
    unsigned remote_port;

    std::time_t start_time;
    context_state_t state{ context_state_t::start };

    prof_accumulator profilers;
    std::map<string, string> custom_log_data;

    boost::atomic_uint64_t bytes_sent; // TODO count
    boost::atomic_uint64_t bytes_received;

    boost::weak_ptr<context> nested_context_;

    void set_request(const request_base_weak_ptr& req)
    {
        current_req_ = req;
    }

    void set_nested(boost::weak_ptr<context> ctx)
    {
        nested_context_ = ctx;
    }

    const char* state_name() const
    {
        switch (state)
        {
        case context_state_t::start:
            return "start";
        case context_state_t::handshake:
            return "handshake";
        case context_state_t::read_headers:
            return "read_headers";
        case context_state_t::read_body:
            return "read_body";
        case context_state_t::execute:
            return "execute";
        default:
            return "unknown";
        };
    }

private:
    ptree_ptr core_get_stat() const override
    {
        ptree_ptr result = yplatform::task_context::core_get_stat();
        result->put(
            "remote_address", remote_address + ":" + boost::lexical_cast<std::string>(remote_port));
        result->put(
            "local_address", local_address + ":" + boost::lexical_cast<std::string>(local_port));
        result->put("lifetime", boost::lexical_cast<std::string>(std::time(nullptr) - start_time));
        result->put("bytes_sent", bytes_sent);
        result->put("bytes_received", bytes_received);
        boost::shared_ptr<context> nested = nested_context_.lock();
        if (nested)
        {
            result->put("current.context", nested->uniq_id());
            request_base_ptr current_req = nested->current_req_.lock();
            if (current_req)
            {
                result->put("current.request", current_req->raw_request_line);
                result->put("current.request", current_req->raw_request_line);
                result->put("current.request", current_req->raw_request_line);
                result->put(
                    "current.transfer-encoding",
                    transfer_encoding_name(current_req->transfer_encoding));
                result->put(
                    "current.content-encoding",
                    content_encoding_name(current_req->content_encoding));
                result->put("current.connection", connection_header_name(current_req->connection));
                if (current_req->upgrade_to != upgrade_none)
                    result->put("current.upgrade", upgrade_proto_name(current_req->upgrade_to));
                result->put("state", nested->state_name());
            }
        }
        return result;
    }

    string transfer_encoding_name(transfer_encoding_header t) const
    {
        if (t == transfer_encoding_identity) return "identity";
        if (t == transfer_encoding_chunked) return "chunked";
        return "ext";
    }

    string content_encoding_name(content_encoding_header t) const
    {
        if (t == content_encoding_identity) return "identity";
        if (t == content_encoding_gzip) return "gzip";
        if (t == content_encoding_compress) return "compress";
        if (t == content_encoding_deflate) return "deflate";
        if (t == content_encoding_all) return "*";
        return "unknown";
    }

    string connection_header_name(connection_header t) const
    {
        if (t == connection_close) return "close";
        if (t == connection_keep_alive) return "keep-alive";
        if (t == connection_upgrade) return "upgrade";
        return "unknown";
    }

    string upgrade_proto_name(upgrade_proto_header t) const
    {
        if (t == upgrade_to_websocket75) return "websocket75";
        if (t == upgrade_to_websocket76) return "websocket76";
        if (t == upgrade_to_websocket07) return "websocket07";
        if (t == upgrade_to_websocket08) return "websocket08";
        if (t == upgrade_to_websocket13) return "websocket13";
        return "unknown";
    }

    request_base_weak_ptr current_req_;
};

typedef boost::shared_ptr<context> context_ptr;

}
