#pragma once

#include <ymod_webserver/types.h>
#include <ymod_webserver/codes.h>
#include <yplatform/exception.h>

namespace ymod_webserver {

class error : public yplatform::exception
{
public:
    inline error() : yplatform::exception()
    {
    }

    explicit inline error(const string& cl) : yplatform::exception(cl, string(), string())
    {
    }

    inline error(const string& cl, string const& pub) : yplatform::exception(cl, pub, string())
    {
    }

    inline error(const string& cl, string const& pub, string const& prv)
        : yplatform::exception(cl, pub, prv)
    {
    }

    inline ~error() noexcept override = default;
};

struct http_error
    : YPLATFORM_ERROR_DEF(http_error, error, "ymod_webserver::http_error", "http_result_error");

struct break_connection
    : YPLATFORM_ERROR_DEF(
          break_connection,
          error,
          "ymod_webserver::break_connection",
          "http_result_error");

struct parse_error
    : YPLATFORM_ERROR_DEF(parse_error, error, "ymod_webserver::parse_error", "parse_error");

struct websocket_error
    : YPLATFORM_ERROR_DEF(
          websocket_error,
          error,
          "ymod_webserver::websocket",
          "websocket_protocol_error");

struct decode_error
    : YPLATFORM_ERROR_DEF(decode_error, error, "ymod_webserver::decode_error", "decode_error");

struct duplicate_error
    : YPLATFORM_ERROR_DEF(
          duplicate_error,
          error,
          "ymod_webserver::duplicate_error",
          "duplicate_error");

struct connection_broken
    : YPLATFORM_ERROR_DEF(
          connection_broken,
          error,
          "ymod_webserver::connection_broken",
          "connection broken");

typedef boost::error_info<struct tag_http_header, ymod_webserver::codes::code> http_result_code;
typedef boost::error_info<struct tag_http_buffer, string> http_result_reason;
// for 3XX error codes:
typedef boost::error_info<struct tag_http_buffer, string> http_error_location;
// for temporary (503) errors:
typedef boost::error_info<struct tag_http_buffer, std::time_t> http_error_retry_after;

template <typename Error, typename ResponsePtr>
void process_exception(const Error& e, ResponsePtr resp)
{
    codes::code result_code = codes::internal_server_error;
    if (const codes::code* c = boost::get_error_info<http_result_code>(e)) result_code = *c;
    if (const string* reason = boost::get_error_info<http_result_reason>(e))
        resp->set_code(result_code, *reason);
    else
        resp->set_code(result_code);
    if (const string* location = boost::get_error_info<http_error_location>(e))
        resp->add_header("Location", *location);
    if (const std::time_t* retry_after = boost::get_error_info<http_error_retry_after>(e))
        resp->add_header("Retry-After", *retry_after);
    resp->result_body(e.public_message());
}

}
