#pragma once

#include <set>
#include <map>
#include <deque>
#include <yplatform/zerocopy/segment.h>
#include <ymod_webserver/types.h>
#include <boost/optional.hpp>

namespace ymod_webserver {

typedef std::map<string, string> header_map_t;

enum transfer_encoding_header
{
    transfer_encoding_identity,
    transfer_encoding_chunked,
    transfer_encoding_ext
};

enum te_encoding_header
{
    te_encoding_none,
    te_encoding_gzip,
    te_encoding_compress,
    te_encoding_deflate
};

enum content_encoding_header
{
    content_encoding_unknown,
    content_encoding_identity,
    content_encoding_gzip,
    content_encoding_compress,
    content_encoding_deflate,
    content_encoding_all
};

enum connection_header
{
    connection_close,
    connection_keep_alive,
    connection_upgrade,
    connection_te
};

enum upgrade_proto_header
{
    upgrade_none,
    upgrade_to_unknown_proto,
    upgrade_to_websocket75,
    upgrade_to_websocket76,
    upgrade_to_websocket07, // http://tools.ietf.org/html/draft-ietf-hybi-thewebsocketprotocol-07
    upgrade_to_websocket08, // http://tools.ietf.org/html/draft-ietf-hybi-thewebsocketprotocol-08
                            // (09, 10, 11, 12)
    upgrade_to_websocket13, // http://tools.ietf.org/html/draft-ietf-hybi-thewebsocketprotocol-13
                            // (14, 15, 16, 17)
    upgrade_to_websocket_unknown
};

enum cache_response_header
{
    cache_response_public,
    cache_response_private,
    cache_response_no_cache,
    cache_response_no_store,
    cache_response_no_transform,
    cache_response_must_revalidate,
    cache_response_proxy_revalidate,
    cache_response_max_age,
    cache_response_s_maxage
};

struct content_description_base
{
    string type;
    string subtype;
};

struct content_description : public content_description_base
{
    content_description() = default;
    std::size_t length{ 0 };
    string boundary; // for multipart body

    bool mime_type_equals(const char* etype, const char* esubtype) const
    {
        return type == etype && subtype == esubtype;
    }

    bool mime_type_equals(const std::string& etype, const std::string& esubtype) const
    {
        return type == etype && subtype == esubtype;
    }
};

struct content_part
{
    header_map_t headers;
    content_description content;
    yplatform::zerocopy::segment body;
    std::vector<content_part> childs;

    const string& header_value(const string& v, const string& def) const
    {
        if (auto i = headers.find(v); i != headers.end()) return i->second;
        return def;
    }

    string header_value(const string& v, const char* def = "") const
    {
        if (auto i = headers.find(v); i != headers.end()) return i->second;
        return def;
    }

    decltype(auto) header_count(string const& v) const
    {
        return headers.count(v);
    }
};

template <typename HeaderValue>
struct header_value_quality
{
    typedef header_value_quality<HeaderValue> this_t;
    header_value_quality() = default;

    HeaderValue value;
    float qvalue{ 1.f };
};

template <typename HV>
struct header_value_quality_less : public std::binary_function<HV, HV, bool>
{
    bool operator()(const HV& x, const HV& y) const
    {
        return x.qvalue < y.qvalue;
    }
};

template <typename HeaderValue>
struct header_with_qvalue
{
    typedef typename std::multiset<
        header_value_quality<HeaderValue>,
        header_value_quality_less<header_value_quality<HeaderValue>>>
        value_t;
};

typedef header_with_qvalue<content_description_base>::value_t accept_header_value_type;
typedef header_with_qvalue<content_encoding_header>::value_t accept_encoding_header_value_type;

typedef boost::optional<accept_header_value_type> accept_header_value;
typedef boost::optional<accept_encoding_header_value_type> accept_encoding_header_value;

}
