#pragma once

namespace ymod_webserver { namespace http {

namespace detail {

// is used in expirable stream after http::stream detach
class fake_streamable
    : public yplatform::net::streamable
    , public boost::enable_shared_from_this<fake_streamable>
{
public:
    void send_client_stream(const yplatform::net::buffers::const_chunk_buffer&) override
    { /* ignore */
    }

    yplatform::net::streamer_wrapper client_stream() override
    {
        return yplatform::net::streamer_wrapper(
            new yplatform::net::streamer<fake_streamable>(shared_from_this()));
    }

    bool is_open() const override
    {
        return false;
    }
};

inline void handle_timer(
    expirable_stream_weak_ptr expirable_stream_weak,
    boost::system::error_code const& err)
{
    auto stream = expirable_stream_weak.lock();
    if (!stream) return;
    if (err != boost::asio::error::operation_aborted)
    {
        if (stream->is_open())
        {
            YLOG_CTX_GLOBAL(stream->ctx(), info)
                << "code=" << codes::gateway_timeout << " message=\"connection control timeout\"";
            stream->send_stored_result();
        }
    }
}

inline void handle_stream_error(
    expirable_stream_weak_ptr expirable_stream_weak,
    boost::system::error_code const& /*err*/)
{
    auto stream = expirable_stream_weak.lock();
    if (!stream) return;
    stream->ctx()->cancel();
    if (stream->is_open())
    {
        YLOG_CTX_GLOBAL(stream->ctx(), info)
            << "code=" << codes::client_closed_connection << " connection closed";
        stream->result(codes::client_closed_connection, "");
    }
}

} // detail

inline expirable_stream::expirable_stream(stream_ptr stream)
    : stream_(stream)
    , req_(stream->request())
    , timer_(stream->make_timer())
    , stored_code_(DEFAULT_TIMEOUT_CODE)
{
}

inline expirable_stream::~expirable_stream()
{
    // @todo explicitly send 500 response lost
    auto stream = detach_stream();
    // no lock for stored_ needed after detaching stream because
    // intermediate result can't be stored after stream is detached
    if (stream && stored_code_ != DEFAULT_TIMEOUT_CODE)
    {
        stream->set_code(stored_code_, stored_reason_);
    }
}

inline stream_ptr expirable_stream::detach_stream()
{
    timer_->cancel();
    boost::mutex::scoped_lock lock(mutex_);
    auto stream_copy = stream_;
    stream_.reset();
    lock.unlock();
    if (stream_copy)
    {
        stream_copy->cancel_poll_connect();
    }
    return stream_copy;
}

inline boost::asio::io_service& expirable_stream::io_service()
{
    return timer_->get_io_service();
}

inline stream_ptr expirable_stream::get_stream() const
{
    boost::mutex::scoped_lock lock(mutex_);
    auto stream_copy = stream_;
    return stream_copy;
}

inline context_ptr expirable_stream::ctx()
{
    return req_->context;
}

inline context_ptr expirable_stream::context()
{
    return req_->context;
}

inline request_ptr expirable_stream::request()
{
    return req_;
}

inline bool expirable_stream::is_open() const
{
    auto stream = get_stream();
    return stream ? stream->is_open() : false;
}

inline void expirable_stream::result(codes::code cd, const string& body)
{
    auto stream = detach_stream();
    if (stream) stream->result(cd, body);
}

inline void expirable_stream::result(codes::code cd, const string& reason, const string& body)
{
    auto stream = detach_stream();
    if (stream)
    {
        stream->set_code(cd, reason);
        stream->result_body(body);
    }
}

inline void expirable_stream::store_intermediate_result(codes::code cd, const string& reason)
{
    boost::mutex::scoped_lock lock(mutex_);
    // only store result if stream is not detached
    if (stream_)
    {
        stored_code_ = cd;
        stored_reason_ = reason;
    }
}

inline void expirable_stream::send_stored_result()
{
    auto stream = detach_stream();
    // no lock for stored_ needed after detaching stream because
    // intermediate result can't be stored after stream is detached
    if (stream) stream->set_code(stored_code_, stored_reason_);
}

inline yplatform::net::streamable_ptr expirable_stream::result_stream(const std::size_t body_len)
{
    static yplatform::net::streamable_ptr fake_streamable(new detail::fake_streamable);
    auto stream = detach_stream();
    return stream ? stream->result_stream(body_len) : fake_streamable;
}

inline yplatform::net::streamable_ptr expirable_stream::result_chunked()
{
    static yplatform::net::streamable_ptr fake_streamable(new detail::fake_streamable);
    auto stream = detach_stream();
    return stream ? stream->result_chunked() : fake_streamable;
}

inline expirable_stream_ptr make_expirable_stream(
    stream_ptr stream,
    const yplatform::time_traits::duration& timeout)
{
    auto result = boost::make_shared<expirable_stream>(stream);
    expirable_stream_weak_ptr result_weak = result;

    result->context()->deadline_from_now(timeout);
    result->timer_->expires_at(result->context()->deadline());
    result->timer_->async_wait(boost::bind(detail::handle_timer, result_weak, _1));
    result->get_stream()->add_error_handler(
        boost::bind(detail::handle_stream_error, result_weak, _1));
    result->get_stream()->begin_poll_connect();
    return result;
}

}}
