#pragma once

namespace ymod_webserver {

#ifndef _LIBCPP_HAS_NO_COROUTINES
template <typename StreamPtr, typename Handler>
inline constexpr bool is_coroutine()
{
    return yplatform::is_awaitable<std::invoke_result_t<Handler, StreamPtr>>();
}
#endif

template <typename StreamPtr, typename Handler>
inline std::function<void(StreamPtr)> adapt_if_coroutine(Handler&& h)
{
#ifndef _LIBCPP_HAS_NO_COROUTINES
    if constexpr (is_coroutine<StreamPtr, Handler>())
    {
        return [h = std::move(h)](auto stream) {
            yplatform::co_spawn(
                stream->get_io_context(),
                [h = std::move(h), s = stream]() mutable { return h(s); },
                [ctx = stream->ctx()](auto exception) {
                    try
                    {
                        if (exception) std::rethrow_exception(exception);
                    }
                    catch (std::exception& e)
                    {
                        LERR_(ctx) << "handler coroutine failed exception=" << e.what();
                    }
                });
        };
    }
    else
#endif
    {
        return std::move(h);
    }
}

template <typename Handler>
void server::bind(const string& ep_name, const std::vector<std::string>& paths, Handler h)
{
    bind_http(ep_name, paths, adapt_if_coroutine<http::stream_ptr>(std::move(h)));
}

template <typename Handler, typename Transformer>
void server::bind(
    const string& ep_name,
    const std::vector<std::string>& paths,
    Handler h,
    Transformer t)
{
    bind_http(
        ep_name,
        paths,
        adapt_if_coroutine<http::stream_ptr>(make_transforming_proxy(std::move(t), std::move(h))));
}

template <typename Handler>
void server::bind_websocket(const string& ep_name, const std::vector<std::string>& paths, Handler h)
{
    bind_websocket(ep_name, paths, adapt_if_coroutine<websocket::stream_ptr>(std::move(h)));
}

template <typename Handler, typename Transformer>
void server::bind_websocket(
    const string& ep_name,
    const std::vector<std::string>& paths,
    Handler h,
    Transformer t)
{
    bind_websocket(
        ep_name,
        paths,
        adapt_if_coroutine<websocket::stream_ptr>(
            make_transforming_proxy(std::move(t), std::move(h))));
}

template <typename Extractor>
void server::set_custom_key_extractor(const string& ep_name, Extractor e)
{
    set_custom_key_extractor(ep_name, http_key_extractor_type(std::move(e)));
}

template <typename Extractor>
void server::set_custom_websocket_key_extractor(const string& ep_name, Extractor e)
{
    set_custom_websocket_key_extractor(ep_name, websocket_key_extractor_type(std::move(e)));
}

template <typename Handler>
void server::set_default_method(const string& ep_name, Handler h)
{
    http_method method = std::move(h);
    set_default_method(ep_name, std::move(method));
}

template <typename Handler>
void server::set_websocket_default_method(const string& ep_name, Handler h)
{
    websocket_method method = std::move(h);
    set_websocket_default_method(ep_name, std::move(method));
}

} // namespace ymod_webserver
