#pragma once

#include <boost/lexical_cast.hpp>
#include <vector>
#include <string>
#include <tuple>
#include <optional>
#include <yplatform/log.h>
#include <cctype>

namespace ymod_webserver { namespace detail {

template <typename T>
struct is_optional : public std::false_type
{
};

template <typename T>
struct is_optional<std::optional<T>> : public std::true_type
{
};

template <typename CheckFunc>
inline bool validate_param(const std::string& param, CheckFunc func)
{
    return std::find_if_not(param.begin(), param.end(), func) == param.end();
}

template <typename T, typename Enable = void>
struct default_validator;

template <typename T>
struct default_validator<T, typename std::enable_if<std::is_unsigned<T>::value>::type>
{
    bool operator()(const std::string& value) const
    {
        return validate_param(value, ::isdigit);
    }
};

template <typename T>
struct default_validator<
    T,
    typename std::enable_if<std::is_signed<T>::value && !std::is_floating_point<T>::value>::type>
{
    bool operator()(const std::string& value) const
    {
        if (value.size())
        {
            return (value[0] == '-' || ::isdigit(value[0])) &&
                std::find_if_not(value.begin() + 1, value.end(), ::isdigit) == value.end();
        }
        return true;
    }
};

template <>
struct default_validator<std::string>
{
    bool operator()(const std::string&) const
    {
        return true;
    }
};

template <typename T>
struct default_validator<T, typename std::enable_if<is_optional<T>::value>::type>
{
    bool operator()(const std::string& value) const
    {
        return impl(value);
    }

    default_validator<typename T::value_type> impl;
};

template <typename F>
struct custom_validator
{
    typename std::decay<F>::type validate_func;

    custom_validator(F&& f) : validate_func(static_cast<F&&>(f))
    {
    }

    bool operator()(const std::string& value) const
    {
        return validate_param(value, validate_func);
    }
};

template <typename T, typename Enable = void>
struct converter
{
    T operator()(const std::string& v) const
    {
        return boost::lexical_cast<T>(v);
    }
};

template <>
struct converter<std::string>
{
    const std::string& operator()(const std::string& v) const
    {
        return v;
    }
};

template <typename T>
struct converter<T, typename std::enable_if<is_optional<T>::value>::type>
{
    auto operator()(const std::string& v) const
    {
        return impl(v);
    }

    converter<typename T::value_type> impl;
};

template <typename T, typename C>
typename std::enable_if<is_optional<T>::value, T>::type convert_value(
    const std::string& value,
    const T& /*default_value*/,
    const C& converter,
    bool /*optional*/)
{
    return converter(value);
}

template <typename T, typename C>
typename std::enable_if<!is_optional<T>::value, T>::type convert_value(
    const std::string& value,
    const T& default_value,
    const C& converter,
    bool optional)
{
    return (optional && value.empty()) ? default_value : converter(value);
}

template <typename T, typename V = default_validator<T>, typename C = converter<T>>
struct argument_base
{
    typedef T value_type;
    typedef V validator_type;
    typedef C converter_type;

    std::string name;
    std::vector<std::string> aliases;
    validator_type validator;
    converter_type converter;
    bool optional;
    value_type default_value;

    bool matches(const std::string& name) const
    {
        return std::find(aliases.begin(), aliases.end(), name) != aliases.end();
    }

    bool validate(const std::string& value) const
    {
        return validator(value);
    }

    /// converter throws
    value_type convert(const std::string& value) const
    {
        return convert_value(value, default_value, converter, optional);
    }
};

} // namespace detail
} // namespace ymod_webserver
