#pragma once

#include <ymod_webserver/methods/detail/arguments.h>
#include <ymod_webserver/methods/detail/tuple_unpack.h>
#include <ymod_webserver/methods/detail/default_answers.h>
#include <string>
#include <tuple>
#include <utility>

namespace ymod_webserver { namespace detail {

struct transform_error
{
    transform_error() = default;

    transform_error(std::string argument_name, std::string reason)
        : argument_name(std::move(argument_name)), reason(std::move(reason))
    {
    }

    operator bool()
    {
        return argument_name.size() || reason.size();
    }

    std::string argument_name;
    std::string reason;
};

template <typename T>
struct extract_validate_error
{
    static const std::string what;
};
template <typename T>
const std::string extract_validate_error<T>::what = "invalid argument";

template <typename T>
struct extract_validate_error<default_validator<T>>
{
    static const std::string what;
};
template <typename T>
const std::string extract_validate_error<default_validator<T>>::what =
    "invalid characters in argument";

template <typename T>
struct extract_validate_error<custom_validator<T>>
{
    static const std::string what;
};
template <typename T>
const std::string extract_validate_error<custom_validator<T>>::what =
    "invalid characters in argument";

template <int index, bool last, typename Params, typename Arguments, typename Results>
struct iterate_tuple
{
    transform_error operator()(const Params& params, const Arguments& arguments, Results& results)
    {
        auto& argument = std::get<index>(arguments);
        auto& result = std::get<index>(results);
        bool found = false;
        for (auto it = params.rbegin(); it != params.rend(); ++it)
        {
            const auto& param = *it;
            if (argument.matches(param.first))
            {
                if (param.second.empty() && !argument.optional)
                {
                    continue;
                }
                found = true;
                if (!argument.validate(param.second))
                {
                    typedef typename std::tuple_element<index, Arguments>::type argument_type;
                    return { param.first,
                             extract_validate_error<typename argument_type::validator_type>::what };
                }
                try
                {
                    result = argument.convert(param.second);
                }
                catch (const std::exception& e)
                {
                    return { param.first, BAD_ARGUMENT_MSG };
                }
                catch (...)
                {
                    return { param.first, BAD_ARGUMENT_MSG };
                }
                break;
            }
        }
        if (!found && !argument.optional)
        {
            return { argument.name, NO_ARGUMENT_MSG };
        }
        else if (!found)
        {
            result = argument.default_value;
        }

        // visit next element
        return iterate_tuple<
            index + 1,
            index + 1 == std::tuple_size<Arguments>::value,
            Params,
            Arguments,
            Results>{}(params, arguments, results);
    }
};

template <int index, typename Params, typename Arguments, typename Results>
struct iterate_tuple<index, true, Params, Arguments, Results>
{
    transform_error operator()(const Params&, const Arguments&, Results&)
    {
        // end of tuple
        return transform_error();
    }
};

template <typename Params, typename Arguments, typename Results>
inline transform_error transform_arguments(
    const Params& params,
    const Arguments& arguments,
    Results& results)
{
    static_assert(
        std::tuple_size<Arguments>::value == std::tuple_size<Results>::value,
        "Requires tuples with the same size");
    return iterate_tuple<0, 0 == std::tuple_size<Arguments>::value, Params, Arguments, Results>{}(
        params, arguments, results);
}

/// Transforms source arguments to data specified types and pass them to a function.
/**
 * Provides the ability to transform source parameter to target
 * data types and call external callback with result arguments.
 * Stores arguments descriptions and reuses it to convert different
 * externally specified params in an absctract map.
 */
template <typename... Args>
class arguments_transformer
{
private:
    typedef std::tuple<typename std::remove_reference<Args>::type...> arguments_type;

    arguments_type arguments;

public:
    typedef typename detail::evaluate_results_tuple<arguments_type>::type results_type;

    arguments_transformer(Args&&... args) : arguments(std::forward<Args>(args)...)
    {
    }

    /// Validates and converts all arguments, returns transform_error
    /// if there are any incorrect arguments.
    template <typename Params>
    transform_error validate(const Params& params) const
    {
        results_type results;
        return transform_arguments(params, arguments, results);
    }

    /// Validates and converts all arguments, returns transform_error
    /// if there are any incorrect arguments.
    template <typename Params>
    transform_error transform(const Params& params, results_type& results) const
    {
        return transform_arguments(params, arguments, results);
    }
};

} // namespace detail
} // namespace ymod_webserver
