#pragma once

#include <ymod_webserver/methods/default_answers.h>
#include <yplatform/log.h>
#include <boost/algorithm/string/case_conv.hpp>
#include <string>
#include <map>

namespace ymod_webserver {

template <typename Stream, typename... Args>
class dispatcher : public yplatform::log::contains_logger
{
public:
    typedef std::function<std::string(Stream)> key_extract_type;
    typedef std::function<void(Stream, Args...)> method_type;
    typedef std::map<std::string, method_type> methods_map_type;

public:
    dispatcher(const key_extract_type& extractor, methods_map_type methods = methods_map_type())
        : key_extractor_(extractor), methods_(methods)
    {
    }

    method_type& operator[](const std::string& key)
    {
        return methods_[key];
    }
    method_type& operator[](const char* key)
    {
        return methods_[key];
    }

    const method_type& operator[](const std::string& key) const
    {
        return methods_[key];
    }
    const method_type& operator[](const char* key) const
    {
        return methods_[key];
    }

    void set_key_extractor(const key_extract_type& extractor)
    {
        key_extractor_ = extractor;
    }

    void set_default_method(const method_type& default_method)
    {
        default_method_ = default_method;
    }

    void operator()(Stream stream, Args&&... args) noexcept
    {
        try
        {
            std::string key = key_extractor_(stream);
            boost::algorithm::to_lower(key);
            stream->request()->bound_path = key;

            auto method = methods_.find(key);
            if (method != methods_.end())
            {
                method->second(stream, std::forward<Args>(args)...);
                return;
            }
            if (default_method_ != nullptr)
            {
                default_method_(stream, std::forward<Args>(args)...);
            }
            else
            {
                default_answers::send_not_found(stream, std::forward<Args>(args)...);
            }
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_LOCAL(stream->ctx(), error)
                << "error while dispatching request: " << ex.what();
        }
        catch (...)
        {
            YLOG_CTX_LOCAL(stream->ctx(), error) << "unexpected error while dispatching request";
        }
    }

    bool empty() const
    {
        return methods_.size() == 0;
    }

private:
    key_extract_type key_extractor_;
    methods_map_type methods_;
    method_type default_method_;
};

}
