#pragma once

#include <ymod_webserver/methods/detail/arguments.h>
#include <ymod_webserver/methods/detail/transform.h>
#include <ymod_webserver/methods/default_answers.h>
#include <vector>
#include <string>
#include <tuple>

namespace ymod_webserver {

template <typename T, typename V = detail::default_validator<T>, typename C = detail::converter<T>>
inline detail::argument_base<T, V, C> argument(
    const std::string& name,
    const std::vector<std::string>& aliases,
    V validator = V(),
    C converter = C())
{
    detail::argument_base<T, V, C> result{ name, aliases, validator, converter, false, T() };
    return result;
}

template <typename T, typename V = detail::default_validator<T>, typename C = detail::converter<T>>
inline detail::argument_base<T, V, C> argument(
    const std::string& name,
    V validator = V(),
    C converter = C())
{
    detail::argument_base<T, V, C> result{ name, { name }, validator, converter, false, T() };
    return result;
}

template <
    typename T,
    typename V = detail::default_validator<T>,
    typename C = detail::converter<T>,
    typename = decltype(std::declval<V>()(std::string()))>
inline detail::argument_base<T, V, C> optional_argument(
    const std::string& name,
    const std::vector<std::string>& aliases,
    T default_value,
    V validator = V(),
    C converter = C())
{
    detail::argument_base<T, V, C> result{
        name, aliases, validator, converter, true, default_value
    };
    return result;
}

template <
    typename T,
    typename V = detail::default_validator<T>,
    typename C = detail::converter<T>,
    typename = decltype(std::declval<V>()(std::string()))>
inline detail::argument_base<T, V, C> optional_argument(
    const std::string& name,
    T default_value,
    V validator = V(),
    C converter = C())
{
    detail::argument_base<T, V, C> result{
        name, { name }, validator, converter, true, default_value
    };
    return result;
}

/// Helper function to create a predefined validator instance.
template <typename T>
inline detail::default_validator<T> default_validator()
{
    return detail::default_validator<T>();
}

/// Helper function to create a custom validator.
/// F has signature: bool f(const std::string&)
template <typename F>
inline detail::custom_validator<F> validator(F&& f)
{
    return detail::custom_validator<F>(static_cast<F&&>(f));
}

/// Helps to create a custom converter.
/// F has signature: T f(const char)
template <typename F>
inline F&& converter(F&& f)
{
    return static_cast<F&&>(f);
}

/// Helps to create an arguments transformer.
template <typename... Args>
inline detail::arguments_transformer<Args...> transformer(Args&&... args)
{
    return detail::arguments_transformer<Args...>(std::forward<Args>(args)...);
}

/// Wraps a handler to apply transformation and call it with converted parameters.
template <typename Transformer, typename Handler>
struct transforming_proxy
{
    using transformer_type = typename std::decay<Transformer>::type;
    typename std::decay<Transformer>::type t;
    Handler h;

    transforming_proxy(Transformer&& t, Handler&& h)
        : t(static_cast<Transformer&&>(t)), h(static_cast<Handler&&>(h))
    {
    }

    template <typename StreamPtr>
    void operator()(StreamPtr stream)
    {
        typename transformer_type::results_type results;
        auto error = t.transform(stream->request()->url.params, results);
        if (!error)
        {
            auto args = std::tuple_cat(std::make_tuple(stream), std::move(results));
            detail::call_with_tuple_args(h, args);
        }
        else
        {
            default_answers::send_bad_request(
                stream, error.reason + " \"" + error.argument_name + "\"");
        }
    }
};

/// Helper for transform_proxy creation.
template <typename T, typename H>
transforming_proxy<T, H> make_transforming_proxy(T&& t, H&& h)
{
    return transforming_proxy<T, H>(static_cast<T&&>(t), static_cast<H&&>(h));
}

} // namespace ymod_webserver