#pragma once

#include <ymod_webserver/codes.h>
#include <ymod_webserver/header.h>
#include <ymod_webserver/request.h>
#include <ymod_webserver/error.h>
#include <yplatform/net/streamable.h>
#include <yplatform/net/types.h>
#include <yplatform/time_traits.h>

namespace ymod_webserver {

class response
    : public yplatform::net::streamable
    , public iabstract
{
public:
    typedef boost::function<void(const boost::system::error_code&)> error_handler;

    virtual yplatform::time_traits::timer_ptr make_timer() const = 0;
    virtual boost::asio::io_service& get_io_service() = 0;
    boost::asio::io_context& get_io_context()
    {
        return get_io_service();
    }

    void result(codes::code cd, const string& body = "")
    {
        set_code(cd);
        result_body(body);
    }

    void result(codes::code cd, const string& reason, const string& body)
    {
        set_code(cd, reason);
        result_body(body);
    }

    virtual void begin_poll_connect() = 0;
    virtual void cancel_poll_connect() = 0;

    virtual void set_code(codes::code cd, const string& reason = "") = 0;
    virtual void add_header(const string& name, const string& value) = 0;
    virtual void add_header(const string& name, std::time_t value) = 0;
    virtual void set_content_type(const string& type, const string& subtype) = 0;
    virtual void set_content_type(const string& type) = 0;
    virtual void set_cache_control(cache_response_header val, const string& ext_val = "") = 0;
    virtual void set_connection(bool close) = 0;

    virtual void result_body(const string& body) = 0;
    virtual yplatform::net::streamable_ptr result_stream(const std::size_t body_len) = 0;
    virtual yplatform::net::streamable_ptr result_chunked() = 0;

    virtual void add_error_handler(const error_handler& handler) = 0;
    virtual void on_error(const boost::system::error_code& e) = 0;
    virtual request_ptr request() const = 0;
    virtual context_ptr ctx() const = 0;

    // call in single thread
    virtual codes::code result_code() const = 0;

    virtual bool is_secure() const = 0;
    virtual const boost::asio::ip::address& remote_addr() const = 0;
};

typedef boost::shared_ptr<response> response_ptr;
typedef boost::weak_ptr<response> response_weak_ptr;

namespace http {
typedef response stream;
typedef boost::shared_ptr<stream> stream_ptr;
typedef boost::weak_ptr<stream> stream_weak_ptr;
}

#define WEB_RESPONSE_LOG(severity, stream_unsafe_param, code, message, logger)                     \
    auto macro_var_stream = stream_unsafe_param;                                                   \
    if (macro_var_stream)                                                                          \
    {                                                                                              \
        logger(macro_var_stream->ctx(), severity)                                                  \
            << "code=" << code << " message=\"" << (message) << "\"";                              \
        macro_var_stream->result(code, (message));                                                 \
    }

#define WEB_RESPONSE_CODE_LOG_G(severity, stream_unsafe_param, code, message)                      \
    WEB_RESPONSE_LOG(severity, stream_unsafe_param, code, message, YLOG_CTX_GLOBAL)

#define WEB_RESPONSE_LOG_L(severity, stream_unsafe_param, code, message)                           \
    WEB_RESPONSE_LOG(                                                                              \
        severity, stream_unsafe_param, ::ymod_webserver::codes::code, message, YLOG_CTX_LOCAL)

#define WEB_RESPONSE_LOG_G(severity, stream_unsafe_param, code, message)                           \
    WEB_RESPONSE_LOG(                                                                              \
        severity, stream_unsafe_param, ::ymod_webserver::codes::code, message, YLOG_CTX_GLOBAL)

#define WEB_RESPONSE_CODE(stream_unsafe_param, code, message)                                      \
    auto macro_var_stream = stream_unsafe_param;                                                   \
    if (macro_var_stream)                                                                          \
    {                                                                                              \
        macro_var_stream->result(code, (message));                                                 \
    }

#define WEB_RESPONSE(stream_unsafe_param, code, message)                                           \
    WEB_RESPONSE_CODE(stream_unsafe_param, ::ymod_webserver::codes::code, message)

#define WEB_RESPONSE_L WEB_RESPONSE_LOG_L

}
