#pragma once

#include <ymod_webserver/settings.h>
#include <ymod_webserver/handler.h>
#include <ymod_webserver/methods/transform.h>
#include <ymod_webserver/methods/default_answers.h>
#include <yplatform/coroutine.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>

namespace ymod_webserver {

/**
 * Provides interface to bind to url paths or specify a whole custom handler.
 */
class server
    : public yplatform::module
    , public boost::noncopyable
{
protected:
    typedef std::function<void(http::stream_ptr)> http_method;
    typedef std::function<void(websocket::stream_ptr)> websocket_method;
    typedef std::function<std::string(http::stream_ptr)> http_key_extractor_type;
    typedef std::function<std::string(websocket::stream_ptr)> websocket_key_extractor_type;

public:
    class impl;

    server(yplatform::reactor& reactor, const yplatform::ptree& conf);
    server(boost::asio::io_service& io, const settings& conf = settings());

    /// Allows to set a custom requests handler.
    void subscribe(const string& ep_name, const handler_ptr& h);

    /**
     * Handler signature: void (stream_ptr)
     */
    template <typename Handler>
    void bind(const string& ep_name, const std::vector<std::string>& paths, Handler h);

    /**
     * Handler with transformer signature:
     * void (stream_ptr, <concrete user types list>)
     */
    template <typename Handler, typename Transformer>
    void bind(
        const string& ep_name,
        const std::vector<std::string>& paths,
        Handler h,
        Transformer t);

    /**
     * Handler signature: void (stream_ptr)
     */
    template <typename Handler>
    void bind_websocket(const string& ep_name, const std::vector<std::string>& paths, Handler h);

    /**
     * Handler with transformer signature:
     * void (stream_ptr, <concrete user types list>)
     */
    template <typename Handler, typename Transformer>
    void bind_websocket(
        const string& ep_name,
        const std::vector<std::string>& paths,
        Handler h,
        Transformer t);

    template <typename Extractor>
    void set_custom_key_extractor(const string& ep_name, Extractor e);

    template <typename Extractor>
    void set_custom_websocket_key_extractor(const string& ep_name, Extractor e);

    template <typename Handler>
    void set_default_method(const string& ep_name, Handler h);

    template <typename Handler>
    void set_websocket_default_method(const string& ep_name, Handler h);

    void start();
    void stop();

    void logger(const yplatform::log::source&); // shadows base class method

    yplatform::ptree get_stats() const override;

private:
    void bind_http(
        const string& ep_name,
        const std::vector<std::string>& paths,
        const http_method& method);
    void bind_websocket(
        const string& ep_name,
        const std::vector<std::string>& paths,
        const websocket_method& method);
    void set_custom_key_extractor(const string& ep_name, http_key_extractor_type&& extractor);
    void set_custom_websocket_key_extractor(
        const string& ep_name,
        websocket_key_extractor_type&& extractor);
    void set_default_method(const string& ep_name, http_method&& method);
    void set_websocket_default_method(const string& ep_name, websocket_method&& method);

    std::shared_ptr<impl> impl_;
};

}

#include <ymod_webserver/impl/server.h>
