#pragma once

#include <ymod_webserver/codes.h>
#include <ymod_webserver/types.h>
#include <ymod_webserver/uri.h>
#include <yplatform/net/settings.h>
#include <yplatform/net/ssl_settings.h>
#include <boost/regex.hpp>
#include <limits>
#include <utility>

namespace ymod_webserver {

struct endpoint
{
    endpoint() = default;
    endpoint(string name, string addr, unsigned short port, bool secure = false)
        : addr(std::move(addr)), port(port), secure(secure), name(std::move(name))
    {
    }
    endpoint(string addr, unsigned short port) : endpoint({}, std::move(addr), port, false)
    {
    }

    void parse_ptree(const yplatform::ptree& conf);

    string addr;
    unsigned short port = 0;
    bool secure = false;
    string name;
    host_info default_host;
    yplatform::net::server_settings socket_settings;
};

struct header_t // TODO rename
{
    std::string original;
    std::string lower;

    bool operator<(const header_t& o) const
    {
        return lower < o.lower;
    }
};

struct rate_limit_settings
{
    struct request_attr
    {
        enum class type
        {
            url_param,
            header
        };

        type type;
        std::string key;

        bool is_url_param() const
        {
            return type == type::url_param;
        }

        bool is_header() const
        {
            return type == type::header;
        }
    };

    struct filter
    {
        request_attr attr;
        boost::regex value;
    };

    string name;
    uint64_t limit = 0;
    uint64_t recovery_rate = 0;
    uint64_t recovery_interval_ms = 0;
    codes::code response_status = codes::service_unavailable;
    std::string response_body;
    std::vector<boost::regex> path;
    std::optional<request_attr> limiting_attr;
    std::vector<filter> filters;

    void parse_limiting_attr(const yplatform::ptree& conf);
    void parse_filters(const yplatform::ptree& conf);
    void parse_ptree(const yplatform::ptree& conf);
};

struct congestion_control_settings
{
    time_traits::duration reactor_overload_delay;

    void parse_ptree(const yplatform::ptree& conf);
};

struct settings
{
    yplatform::net::ssl_settings ssl;
    std::multimap<string, endpoint> endpoints;
    std::vector<rate_limit_settings> rate_limits;

    boost::optional<yplatform::log::source> access_log;
    boost::optional<yplatform::log::typed::logger> access_log_tskv;
    string policy_file;
    unsigned read_chunk_size;
    unsigned max_request_line_size;
    unsigned max_headers_size;
    unsigned max_post_size;
    unsigned max_websocket_size;
    bool enable_gzip;
    bool enable_deflate;
    bool enable_compress;
    string websocket_origin;
    string websocket_location;
    string websocket_protocol;
    bool connect_per_request;
    bool enable_websocket_masking;
    bool log_cookies;
    size_t ws_max_length;
    size_t ws_max_fragmentation;
    size_t ws_max_streams;
    struct
    {
        unsigned char major;
        unsigned char minor;
    } http_default_version;
    std::vector<string> protected_url_params;
    std::set<header_t> log_headers;
    bool access_log_tskv_socket_stats;
    bool access_log_tskv_timings;
    unsigned keep_alive_requests;
    string access_control_allow_origin;
    string access_control_allow_headers;
    string access_control_expose_headers;
    bool access_control_allow_credentials;
    yplatform::time_traits::duration min_acceptable_timeout;
    bool augment_unique_id_with_request_id;
    boost::optional<congestion_control_settings> congestion_control;
    bool enable_timing_stats;

    settings();
    settings(endpoint endpoint);

    void load_policy_data(const string& policy_file_name);
    void parse_ptree(const yplatform::ptree& data);
};

}
