#pragma once

#include <map>
#include <deque>
#include <ymod_webserver/types.h>
#include <boost/lexical_cast.hpp>
#include <boost/range/iterator_range_core.hpp>

namespace ymod_webserver {

typedef std::multimap<string, string> param_map_t;
typedef std::deque<string> uri_path_t;

struct host_info
{
    host_info() : proto("http")
    {
    }

    string proto;
    string domain;
    unsigned port{ 80 };

    // <0 if less, 0 if equal and >0 if great
    static int compare(const host_info& h1, const host_info& h2)
    {
        if (h1.port != h2.port) return (h1.port < h2.port ? -1 : 1);
        int cmp_res = h1.domain.compare(h2.domain);
        if (cmp_res) return cmp_res;
        return h1.proto.compare(h2.proto);
    }

    bool empty() const
    {
        return domain.empty();
    }

    bool operator==(const host_info& info) const
    {
        return compare(*this, info) == 0;
    }

    bool operator!=(const host_info& info) const
    {
        return compare(*this, info) != 0;
    }

    bool operator<(const host_info& info) const
    {
        return compare(*this, info) < 0;
    }

    bool operator<=(const host_info& info) const
    {
        return compare(*this, info) <= 0;
    }

    bool operator>(const host_info& info) const
    {
        return compare(*this, info) > 0;
    }

    bool operator>=(const host_info& info) const
    {
        return compare(*this, info) >= 0;
    }
};

struct http_uri
{
    host_info host;
    uri_path_t path;
    string fragment;
    param_map_t params;

    const string& param_value(const string& v, const string& def) const
    {
        auto i = params.find(v);
        if (i != params.end()) return i->second;
        return def;
    }

    string param_value(const string& v, const char* def = "") const
    {
        auto i = params.find(v);
        if (i != params.end()) return i->second;
        return def;
    }

    param_map_t::const_iterator param_find(string const& v) const
    {
        return params.find(v);
    }

    boost::iterator_range<param_map_t::const_iterator> param_values(string const& v) const
    {
        std::pair<param_map_t::const_iterator, param_map_t::const_iterator> p =
            params.equal_range(v);

        return boost::iterator_range<param_map_t::const_iterator>(p.first, p.second);
    }

    param_map_t::size_type param_count(string const& v) const
    {
        return params.count(v);
    }

    template <typename T>
    const T param_value_cast(const string& v, const T& def) const
    {
        auto i = params.find(v);
        if (i != params.end())
        {
            try
            {
                return boost::lexical_cast<T>(i->second);
            }
            catch (const boost::bad_lexical_cast& e)
            {
                throw std::runtime_error("can't cast parameter " + v);
            }
        }
        return def;
    }

    string make_full_path() const
    {
        string result = "/";
        auto it = path.begin();
        while (it != path.end())
        {
            result.append(*it);
            ++it;
            if (it != path.end()) result.append("/");
        }
        return result;
    }
};

void make_uri(const string& path, http_uri& u);

} // ymod_webserver
