#pragma once

#include <ymod_webserver/types.h>
#include <ymod_webserver/websocket_body.h>
#include <ymod_webserver/websocket_codes.h>
#include <ymod_webserver/iabstract.h>
#include <ymod_webserver/request.h>
#include <ymod_webserver/error.h>
#include <yplatform/future/future.hpp>
#include <yplatform/zerocopy/segment.h>
#include <yplatform/net/types.h>
#include <yplatform/net/streamable.h>
#include <yplatform/time_traits.h>

namespace ymod_webserver { namespace websocket {

struct message
{
    enum
    {
        bit_fin = 0x80,
        bit_fin_val = 0x80,
        bit_rsv1 = 0x40,
        bit_rsv1_val = 0x40,
        bit_rsv2 = 0x20,
        bit_rsv2_val = 0x20,
        bit_rsv3 = 0x10,
        bit_rsv3_val = 0x10,
        bit_mask = 0x08,
        bit_mask_val = 0x80
    };
    enum
    {
        opcode_continuation = 0x00,
        opcode_text = 0x01,
        opcode_binary = 0x02,
        // 0x03-0x07 reserved for further non-control frames
        opcode_close = 0x08,
        opcode_ping = 0x09,
        opcode_pong = 0xA
        // 0x0B-0x0F reserved for further non-control frames
    };

    message()
    {
    }

    bool is_finished() const
    {
        return bits & bit_fin_val;
    }

    uint8_t bits{ bit_fin };
    uint8_t opcode{ 0 };
    mask_t mask;
    uint64_t length{ 0 };
    yplatform::zerocopy::segment ext;
    payload data;
};

/**
 * Control state for fragmented messages
 **/
struct fragmented_message_state
{

    fragmented_message_state(const message& msg)
        : is_finished(false), opcode(msg.opcode), length(msg.length), fragments_count(1)
    {
    }

    fragmented_message_state& operator<<(const message& msg)
    {
        is_finished = msg.is_finished();
        length += msg.length;
        fragments_count++;
        return *this;
    }

    bool is_finished;
    uint8_t opcode;
    uint64_t length;
    uint64_t fragments_count;
};

typedef yplatform::future::future<void> future_void_t;

// TODO rename (not 'output') but keep compartibility
class output_stream
    : public yplatform::net::streamable
    , public iabstract
{
public:
    typedef boost::function<void(websocket::message const& msg)> message_hook_t;
    typedef boost::function<void(uint16_t code, const string& reason)> close_hook_t;

    virtual yplatform::time_traits::timer_ptr make_timer() const = 0;
    virtual boost::asio::io_service& get_io_service() = 0;

    virtual void begin_receive(
        const yplatform::time_traits::duration& inactive_timeout =
            yplatform::time_traits::duration::max()) = 0;

    virtual future_void_t send_text(const string& msg, uint8_t opcode = message::opcode_text) = 0;
    virtual future_void_t send_binary(
        const string& msg,
        uint8_t opcode = message::opcode_binary) = 0;
    virtual void close_connection(
        uint16_t code = codes::close_opcode_normal,
        const string& reason = "") = 0;

    virtual yplatform::net::streamer_wrapper text_stream(
        std::size_t sz,
        uint8_t opcode = message::opcode_text) = 0;
    virtual yplatform::net::streamer_wrapper bin_stream(
        std::size_t sz,
        uint8_t opcode = message::opcode_binary) = 0;
    yplatform::net::streamer_wrapper client_stream() override
    {
        throw std::runtime_error("not implemented");
    }

    virtual request_ptr request() const = 0;

    context_ptr ctx() const
    {
        return request()->context;
    }

    virtual void add_message_callback(message_hook_t const& hook) = 0;
    virtual void set_close_callback(close_hook_t const& hook) = 0;

    virtual bool is_secure() const = 0;
    bool is_open() const override = 0;
};

typedef boost::shared_ptr<output_stream> output_stream_ptr;

typedef output_stream stream;
typedef boost::shared_ptr<stream> stream_ptr;
typedef boost::weak_ptr<stream> stream_weak_ptr;

}}
