#pragma once

#include <ymod_webserver/websocket.h>
#include <ymod_webserver/websocket_body.h>

namespace ymod_webserver { namespace websocket {

template <typename Stream>
std::size_t make_websocket_package_header(
    uint8_t bits,
    uint8_t opcode,
    const mask_t& mask,
    uint64_t length,
    Stream& stream)
{
    if (bits & message::bit_fin) opcode |= message::bit_fin_val;
    if (bits & message::bit_rsv1) opcode |= message::bit_rsv1_val;
    if (bits & message::bit_rsv2) opcode |= message::bit_rsv2_val;
    if (bits & message::bit_rsv3) opcode |= message::bit_rsv3_val;
    stream << static_cast<char>(opcode);

    uint8_t payload_len;
    if (length > 65535) payload_len = 127;
    else if (length > 125)
        payload_len = 126;
    else
        payload_len = static_cast<uint8_t>(length);
    if (bits & message::bit_mask) payload_len |= message::bit_mask_val;
    stream << static_cast<char>(payload_len);

    std::size_t message_length = 2;
    if (length > 125)
    {
        char* length_bytes = reinterpret_cast<char*>(&length);
        // message length can't be packed in one byte, there are two cases: 2 bytes or 8 bytes
        int top = (length > 65535) ? 8 : 2;
        for (int i = top; i > 0; i--)
            stream << length_bytes[i - 1];
        message_length += top;
    }
    if (bits & message::bit_mask)
    {
        stream << mask[0] << mask[1] << mask[2] << mask[3];
        message_length += 4;
    }
    return message_length;
}

template <typename Stream>
std::size_t prepare_ws_message(
    const string& msg,
    uint8_t opcode,
    uint8_t bits,
    const mask_t& mask,
    bool apply_mask,
    Stream& stream)
{
    if (mask != empty_mask) bits |= message::bit_mask;
    std::size_t header_length =
        make_websocket_package_header(message::bit_fin, opcode, mask, msg.size(), stream);
    if (apply_mask)
    {
        typedef mask_iterator<string::const_iterator> mask_iter;
        stream << boost::make_iterator_range(
            mask_iter(msg.begin(), mask), mask_iter(msg.end(), mask));
    }
    else
    {
        stream << msg;
    }
    return header_length + msg.length();
}

}}