#pragma once

#include "log_util.h"
#include "protect_cookies.h"
#include "protect_url_params.h"
#include <ymod_webserver/settings.h>

namespace ymod_webserver {

inline const std::string& get_or_empty(const header_map_t& headers, const std::string& name)
{
    static const std::string empty;
    const auto it = headers.find(name);
    return it == headers.end() ? empty : it->second;
}

template <typename StreamPtr>
void write_2_access_log(std::shared_ptr<settings> settings, StreamPtr stream_ptr)
{
    static const std::string HOST("host");
    static const std::string REFERER("referer");
    static const std::string USER_AGENT("user-agent");
    static const std::string COOKIE("cookie");
    static const std::string X_FORWARDED_FOR("x-forwarded-for");

    auto session = stream_ptr->get_session();
    auto& req = *stream_ptr->request();
    auto ip = session->remote_addr().to_string();
    auto status_code = stream_ptr->result_code();
    bool secure = session->is_secure();
    auto proceed_time = time_traits::duration_cast<time_traits::milliseconds>(
                            time_traits::clock::now() - stream_ptr->created_at())
                            .count();
    auto ctx = req.context;
    std::string cookies;
    if (settings->log_cookies)
    {
        cookies = get_or_empty(req.headers, COOKIE);
        protect_sensitive_cookies(cookies);
    }
    auto& host = get_or_empty(req.headers, HOST);
    auto& referer = get_or_empty(req.headers, REFERER);
    auto& user_agent = get_or_empty(req.headers, USER_AGENT);
    auto& forwarded_for = get_or_empty(req.headers, X_FORWARDED_FOR);
    auto& request_line = req.raw_request_line;

    if (settings->access_log)
    {
        std::ostringstream os;
        log_access_time(os);
        protect_url_params(request_line, settings->protected_url_params);
        os << host << " " << ip << " \"" << request_line << "\" " << status_code << " \"" << referer
           << "\" \"" << user_agent << "\" \"" << cookies << "\" \"" << forwarded_for << "\" "
           << (secure ? "TLS" : "-") << " \"" << ctx->uniq_id() << "\"";
        log_headers(*settings, req.headers, os);
        os << ' ' << (proceed_time / 1000) << "." << std::setw(3) << std::setfill('0')
           << (proceed_time % 1000);
        YLOG((*settings->access_log), info) << os.str();
    }

    if (settings->access_log_tskv)
    {
        using namespace yplatform::log::typed;
        attributes_map extra_attrs;
        log_headers(*settings, req.headers, extra_attrs);
        log_custom_data(*ctx, extra_attrs);
        if (settings->access_log_tskv_socket_stats)
        {
            auto socket_stats = session->stats().socket_info;
            if (socket_stats.size()) extra_attrs << make_attr("socket_stats", socket_stats);
        }
        if (settings->access_log_tskv_timings)
        {
            log_profilers_data(*ctx, extra_attrs);
        }
        protect_url_params(req.raw_url, settings->protected_url_params);
        YLOG((*settings->access_log_tskv), info)
            << make_attr("host", host) << make_attr("connection", session->ctx()->uniq_id())
            << make_attr("ip", ip) << make_attr("method", parser::get_method_name(req.method))
            << make_attr("request", req.raw_url) << make_attr("status_code", status_code)
            << make_attr("referer", referer) << make_attr("user_agent", user_agent)
            << make_attr("cookies", cookies) << make_attr("forwarded_for", forwarded_for)
            << make_attr("secure", secure) << make_attr("y_context", ctx->uniq_id()) << extra_attrs;
    }
}

}
