#pragma once
#include <ymod_webserver/settings.h>
#include <ymod_webserver/types.h>
#include <yplatform/algorithm/congestion_controller.h>
#include <yplatform/reactor.h>
#include <yplatform/reactor/tracer.h>
#include <boost/core/noncopyable.hpp>
#include <mutex>

namespace ymod_webserver {

class connection_controller
{
public:
    connection_controller(
        yplatform::reactor_ptr reactor,
        const congestion_control_settings& settings)
        : reactor_name_(reactor->name()), settings_(settings)
    {
        for (size_t i = 0; i < reactor->size(); ++i)
        {
            controllers_.emplace(
                (*reactor)[i]->io(), std::make_shared<yplatform::congestion_controller>());
        }

        for (size_t i = 0; i < reactor->size(); ++i)
        {
            auto& io = *(*reactor)[i]->io();
            auto tracer = std::make_shared<yplatform::reactor_tracer>(io);
            store_tracer(io, tracer);
            tracer->start();
        }
    }

    bool try_run(boost::asio::io_service* io)
    {
        auto tracer = yplatform::find_tracer(*io);
        auto controller = find_controller(io);

        if (!tracer || !controller) return true;

        std::scoped_lock lock(mux_);
        return controller->try_run();
    }

    void finish(boost::asio::io_service* io)
    {
        auto tracer = yplatform::find_tracer(*io);
        auto controller = find_controller(io);

        if (!tracer || !controller) return;

        std::scoped_lock lock(mux_);
        if (tracer->last_delay() > settings_.reactor_overload_delay)
        {
            controller->finish_with_congestion();
        }
        else
        {
            controller->finish_without_congestion();
        }
    }

    yplatform::ptree get_stats() const
    {
        yplatform::ptree ret;

        std::size_t i = 0;
        for (auto&& [_, controller] : controllers_)
        {
            std::scoped_lock lock(mux_);
            ret.push_back(std::pair(reactor_name_ + std::to_string(i), controller->stats()));
            ++i;
        }
        return ret;
    }

private:
    yplatform::congestion_controller_ptr find_controller(boost::asio::io_service* io)
    {
        auto it = controllers_.find(io);
        return it != controllers_.end() ? it->second : nullptr;
    }

    string reactor_name_;
    congestion_control_settings settings_;
    mutable std::mutex mux_;
    std::unordered_map<boost::asio::io_service*, yplatform::congestion_controller_ptr> controllers_;
};

}
