#include "http_stream.h"
#include "validator.h"
#include "version.h"
#include <yplatform/app_service.h>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/assign/list_of.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include "parser/date.h"
#include "net_server.h"

namespace ymod_webserver {

namespace {
inline void do_nothing(boost::shared_ptr<http_stream> /*starter*/)
{
}
void on_error_hook(boost::weak_ptr<http_stream> stream_weak, boost::system::error_code const& err)
{
    if (boost::shared_ptr<http_stream> stream_shared = stream_weak.lock())
        stream_shared->on_error(err);
}
}

const std::vector<string> http_stream::cache_control_header_values =
    boost::assign::list_of("public")("private")("no-cache")("no-store")("no-transform")(
        "must-revalidate")("proxy-revalidate")("max-age")("s-maxage");

http_stream::http_stream(
    boost::asio::io_service& io,
    boost::weak_ptr<net_server> owner,
    context_ptr context,
    net_session_ptr session,
    const request_ptr& req,
    read_buffer_ptr readq)
    : io_(&io)
    , owner_(owner)
    , context_(context)
    , session_(session)
    , request_(req)
    , readq_(readq)
    , i_saved_(readq_->begin())
    , state_(send_response_line)
    , result_code_(codes::internal_server_error)
    , send_queue_size_(0)
    , content_type_sent_(false)
    , access_control_allow_origin_sent_(false)
    , connection_close_(session->session_settings().connect_per_request)
{
    ctx()->profilers.push("exec");
    yplatform::get_context_repository(*io_).add_context(ctx());
}

http_stream::~http_stream()
{
    try
    {
        yplatform::get_context_repository(*io_).rem_context(ctx());
        if (state_ == send_response_line) result(codes::internal_server_error);
        else
        {
            if (state_ == send_headers)
            {
                result_body("");
            }
            else if (state_ == send_queue)
            {
                send_response_queue();
            }
            else if (state_ == send_chunked)
            {
                session_->client_stream() << "0\r\n\r\n";
            }
        }
        ctx()->profilers.pop("exec");
    }
    catch (...)
    {
    }
    if (boost::shared_ptr<net_server> powner = owner_.lock()) try
        {
            powner->on_destroy(this);
        }
        catch (...)
        {
        }
}

void http_stream::init()
{
    session_->set_write_error_hook(boost::bind(on_error_hook, weak_from_this(), _1));
}

void http_stream::send_response_headers(int64_t s)
{
    assert(state_ == send_headers);
    auto& settings = session_->session_settings();
    if (s < 0 && (request_->proto_version.first == 0 || request_->proto_version.second == 0))
    {
        state_ = send_queue;
        return;
    }
    if (request_->proto_version.first == 0) return;

    (*out_stream_) << "Y-Context: " << request_->context->uniq_id() << "\r\n";

    if (connection_close_) (*out_stream_) << "Connection: close\r\n";
    else if (is_keep_alive())
        (*out_stream_) << "Connection: keep-alive\r\n";
    else
        (*out_stream_) << "Connection: close\r\n";

    if (!content_type_sent_) (*out_stream_) << "Content-Type: text/html\r\n";

    if (!access_control_allow_origin_sent_ && settings.access_control_allow_origin.size())
    {
        (*out_stream_) << "Access-Control-Allow-Origin: " << settings.access_control_allow_origin
                       << "\r\n";
        if (settings.access_control_expose_headers.size())
            (*out_stream_) << "Access-Control-Expose-Headers: "
                           << settings.access_control_expose_headers << "\r\n";
        if (settings.access_control_allow_credentials)
            (*out_stream_) << "Access-Control-Allow-Credentials: true\r\n";
    }

    if (s >= 0) (*out_stream_) << "Content-Length: " << s << "\r\n\r\n";
    else
    {
        if (request_->proto_version.first > 0 && request_->proto_version.second > 0)
        {
            (*out_stream_) << "Transfer-Encoding: chunked\r\n\r\n";
            state_ = send_chunked;
        }
        else
        {
            state_ = send_queue;
        }
    }
}

void http_stream::send_response_queue()
{
    state_ = send_headers;
    send_response_headers(send_queue_size_);
    out_stream_.reset();
    typedef std::deque<yplatform::net::buffers::const_chunk_buffer>::const_iterator iterator_t;
    for (iterator_t i_buff = send_queue_.begin(), i_buff_end = send_queue_.end();
         i_buff != i_buff_end;
         ++i_buff)
    {
        session_->send_client_stream(*i_buff);
    }
}

void http_stream::set_code(codes::code cd, const string& reason)
{
    if (state_ != send_response_line) return;
    yplatform::net::streamer_wrapper tmp_wrapper = session_->client_stream();
    out_stream_.reset(new yplatform::net::streamer_wrapper(std::move(tmp_wrapper)));
    result_code_ = cd;
    state_ = send_headers;
    reason_ = (reason.empty() ? ymod_webserver::codes::reason::get(cd) : reason);
    if (request_->proto_version.first == 0) return;
    (*out_stream_) << "HTTP/" << static_cast<int>(request_->proto_version.first) << "."
                   << static_cast<int>(request_->proto_version.second) << " " << cd << " "
                   << reason_ << "\r\n"
                   << "Server: YaWEBServer/" << VERSION << "\r\n"
                   << "Date: ";
    parser::print_date(*out_stream_, std::time(0));
    (*out_stream_) << "\r\n";
}

void http_stream::add_header(const string& name, const string& value)
{
    assert(state_ == send_headers);
    using boost::iequals;
    if (request_->proto_version.first == 0) return;
    if (iequals(name, "Content-Type"))
    {
        content_type_sent_ = true;
    }
    if (iequals(name, "Access-Control-Allow-Origin"))
    {
        access_control_allow_origin_sent_ = true;
    }
    (*out_stream_) << name << ": " << value << "\r\n";
}

void http_stream::add_header(const string& name, std::time_t value)
{
    assert(state_ == send_headers);
    if (request_->proto_version.first == 0) return;
    (*out_stream_) << name << ": ";
    parser::print_date(*out_stream_, value);
    (*out_stream_) << "\r\n";
}

void http_stream::result_body(const string& body)
{
    const string& body_impl = (body.empty() ? reason_ : body);
    send_response_headers(body_impl.length());
    state_ = send_body;
    (*out_stream_) << body_impl;
    out_stream_.reset();
}

yplatform::net::streamable_ptr http_stream::result_stream(const std::size_t body_len)
{
    send_response_headers(body_len);
    out_stream_.reset();
    state_ = send_body;
    return shared_from_this();
}

yplatform::net::streamable_ptr http_stream::result_chunked()
{
    send_response_headers(-1);
    if (state_ == send_chunked) out_stream_.reset();
    return shared_from_this();
}

bool http_stream::is_keep_alive() const
{
    return !connection_close_ && request()->connection == connection_keep_alive &&
        session_->requests_count() < session_->session_settings().keep_alive_requests
        // we haven't read all request, so must close connection
        && result_code_ != codes::request_entity_too_large;
}

void http_stream::send_client_stream(yplatform::net::buffers::const_chunk_buffer const& s)
{
    if (state_ == send_queue)
    {
        send_queue_.push_back(s);
        send_queue_size_ += s.size();
        return;
    }
    if (s.size() == 0) return;
    if (state_ == send_chunked)
    {
        session_->client_stream() << std::hex << s.size() << "\r\n";
        session_->send_client_stream(s);
        session_->client_stream() << "\r\n";
    }
    else
    {
        session_->send_client_stream(s);
    }
}

void http_stream::begin_poll_connect()
{
    session_->begin_read(
        boost::protect(
            boost::bind(&http_stream::handle_poll_connect, this->shared_from_this(), _1)),
        readq_->prepare(1),
        1);
}

void http_stream::handle_poll_connect(const boost::system::error_code& e)
{
    if (e)
    {

        if (connection_is_closed(e))
        {
            // ignore
        }
        else if (e == boost::asio::error::operation_aborted)
        {
            // supposed to be canceled manually using session_->cancel_operations();
            return;
        }
        else
        {
            YLOG_CTX_LOCAL(request_->context, info)
                << "handle_poll_connect error message=\"" << e.message() << "\"";
            if (session_->is_open())
                session_->async_close(boost::bind(do_nothing, shared_from_this()));
        }

        on_error(e);
        return;
    }
    begin_poll_connect();
}

}
