#pragma once

#include <ymod_webserver/codes.h>
#include <ymod_webserver/request.h>
#include <ymod_webserver/response.h>
#include "contains_creation_time.h"
#include "session.h"

namespace ymod_webserver {

class net_server;

class http_stream
    : public response
    , public contains_creation_time
    , public boost::enable_shared_from_this<http_stream>
{
    static const std::vector<string> cache_control_header_values;
    enum send_state
    {
        send_response_line,
        send_headers,
        send_body,
        send_chunked,
        send_queue
    };
    typedef read_buffer_t::iterator read_iterator;

public:
    http_stream(
        boost::asio::io_service& io,
        boost::weak_ptr<net_server> owner,
        context_ptr context,
        net_session_ptr session,
        const request_ptr& req,
        read_buffer_ptr readq);

    virtual ~http_stream();

    void init();

    void set_code(codes::code cd, const string& reason = "") override;
    void add_header(const string& name, const string& value) override;
    void add_header(const string& name, std::time_t value) override;

    void result_body(const string& body) override;
    yplatform::net::streamable_ptr result_stream(const std::size_t body_len) override;
    yplatform::net::streamable_ptr result_chunked() override;

    yplatform::net::streamer_wrapper client_stream() override
    {
        return yplatform::net::streamer_wrapper(
            new yplatform::net::streamer<http_stream>(shared_from_this()));
    }

    bool is_open() const override
    {
        return session_->is_open();
    }

    bool is_secure() const override
    {
        return session_->is_secure();
    }

    void set_content_type(const string& type, const string& subtype) override
    {
        set_content_type(type + "/" + subtype);
    }

    void set_content_type(const string& type) override
    {
        add_header("Content-Type", type);
    }

    void set_cache_control(cache_response_header val, const string& ext_val) override
    {
        if (ext_val.empty()) add_header("Cache-Control", cache_control_header_values.at(val));
        else
            add_header("Cache-Control", cache_control_header_values.at(val) + "=" + ext_val);
    }

    void set_connection(bool close) override
    {
        connection_close_ = close;
    }

    bool is_keep_alive() const;

    void send_client_stream(yplatform::net::buffers::const_chunk_buffer const& s) override;

    yplatform::time_traits::timer_ptr make_timer() const override
    {
        return std::make_shared<yplatform::time_traits::timer>(session_->current_io_service());
    }

    boost::asio::io_service& get_io_service() override
    {
        return session_->current_io_service();
    }

    void add_error_handler(const error_handler& handler) override
    {
        error_handlers_.push_back(handler);
        if (!session_->is_open())
        {
            handler(boost::system::error_code());
        }
    }

    void on_error(const boost::system::error_code& e) override
    {
        for (auto& handler : error_handlers_)
        {
            handler(e);
        }
    }

    void begin_poll_connect() override;

    void cancel_poll_connect() override
    {
        auto _this = this->shared_from_this();
        session_->cancel_operations([_this]() {});
    }

    context_ptr ctx() const override
    {
        return context_;
    }

    // call in single thread
    codes::code result_code() const override
    {
        return result_code_;
    }

    const boost::asio::ip::address& remote_addr() const override
    {
        return session_->remote_addr();
    }

    request_ptr request() const override
    {
        return request_;
    }

    net_session_ptr get_session()
    {
        return session_;
    }

protected:
    void send_response_headers(int64_t s);
    void send_response_queue();

    void handle_poll_connect(const boost::system::error_code& e);

    boost::weak_ptr<http_stream> weak_from_this()
    {
        return shared_from_this();
    }

    boost::asio::io_service* io_ = nullptr;
    boost::weak_ptr<net_server> owner_;
    context_ptr context_;
    net_session_ptr session_;
    request_ptr request_;
    read_buffer_ptr readq_;
    read_iterator i_saved_;
    send_state state_;
    codes::code result_code_;
    string reason_;
    std::deque<yplatform::net::buffers::const_chunk_buffer> send_queue_;
    std::size_t send_queue_size_;
    bool content_type_sent_;
    bool access_control_allow_origin_sent_;
    bool connection_close_;
    std::unique_ptr<yplatform::net::streamer_wrapper> out_stream_;
    std::deque<error_handler> error_handlers_;

    boost::mutex mutex_;
};

}
