#include <ymod_webserver/settings.h>
#include <ymod_webserver/context.h>
#include <yplatform/log/typed.h>
#include <sstream>

namespace ymod_webserver {

inline void log_access_time(std::ostringstream& os)
{
    static const char* month_names[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
                                         "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    std::time_t tnow = std::time(0);
    std::tm now;
    localtime_r(&tnow, &now);
    os << "[" << std::setw(2) << std::setfill('0') << now.tm_mday << "/" << month_names[now.tm_mon]
       << "/" << std::setw(4) << (1900 + now.tm_year) << ":" << std::setw(2) << now.tm_hour << ":"
       << std::setw(2) << now.tm_min << ":" << std::setw(2) << now.tm_sec << " " << now.tm_zone
       << "] ";
}

inline void log_headers(
    const settings& settings,
    const header_map_t& all_headers,
    std::ostringstream& os)
{
    if (settings.log_headers.empty()) return;
    os << " \"";
    for (const auto& header : settings.log_headers)
    {
        const auto it = all_headers.find(header.lower);
        if (it != all_headers.end())
        {
            os << header.original << ": " << it->second << ", ";
        }
    }
    os << '"';
}

inline string replace_column_name_chars(const string& name)
{
    const char SPECIAL = '_';
    string ret;
    ret.reserve(name.size());
    bool beginning = true;
    bool escaped = false;
    for (auto& ch : name)
    {
        // Special char and digits in the beginning are denied.
        if (beginning && (!isalpha(ch))) continue;
        beginning = false;
        if (isalnum(ch))
        {
            ret += ch;
            escaped = false;
            continue;
        }
        if (!escaped)
        {
            ret += SPECIAL;
            // Column name can't contain special symbol sequences.
            escaped = true;
        }
    }
    return ret;
}

inline void log_headers(
    const settings& settings,
    const header_map_t& all_headers,
    yplatform::log::typed::attributes_map& os)
{
    using yplatform::log::typed::make_attr;
    for (const auto& header : settings.log_headers)
    {
        const auto it = all_headers.find(header.lower);
        if (it != all_headers.end())
        {
            auto key = replace_column_name_chars(header.lower);
            if (key.size()) os << make_attr(key, it->second);
        }
    }
}

inline void log_custom_data(const context& ctx, yplatform::log::typed::attributes_map& os)
{
    using yplatform::log::typed::make_attr;
    for (auto& pair : ctx.custom_log_data)
    {
        auto key = replace_column_name_chars(pair.first);
        if (key.size()) os << make_attr(key, pair.second);
    }
}

inline void log_profilers_data(const context& ctx, yplatform::log::typed::attributes_map& os)
{
    using yplatform::log::typed::make_attr;
    for (auto& v : ctx.profilers.get_values())
    {
        os << make_attr(
            "profiler_" + replace_column_name_chars(v.name), prof_time_to_string(v.time));
    }
    os << make_attr("profiler_total", prof_time_to_string(ctx.profilers.get_total()));
}

}
