#include "net_server.h"
#include "access_log.h"
#include "session.h"
#include "starter.h"
#include "websocket_stream.h"
#include "http_stream.h"

#include <ymod_webserver/methods/default_answers.h>
#include <cmath>

namespace ymod_webserver {

#define NETSERVER_LOG(severity)                                                                    \
    YLOG_L(severity) << "object=\"netserver\" function=\"" << __FUNCTION__ << "\" "

#define NETSERVER_CTX_LOG(ctx, severity)                                                           \
    YLOG_L(severity) << "ctx=" << (ctx ? ctx->uniq_id() : string("unknown"))                       \
                     << " object=netserver function=" << __FUNCTION__ << " "

namespace {
void notify_request_destroyed(net_server_weak_ptr netserver, request* req)
{
    if (net_server_ptr pserver = netserver.lock()) try
        {
            pserver->on_destroy(req);
        }
        catch (...)
        {
        }
}

inline double exec_time_in_sec(const prof_accumulator& profilers)
{
    for (auto&& [name, time] : profilers.get_values())
    {
        if (name == "exec") 
        {
            return static_cast<double>(time) / 1000;
        }
    }
    return NAN;
}
}

void yplatform_net_server::register_session(session_ptr session)
{
    // TODO try not to use context_repo - wait in net_server
    session->set_owner(weak_from_this());

    on_session_ready(session);
}

void yplatform_net_server::on_session_ready(net_session_ptr session)
{
    session->increment_requests_count();
    boost::shared_ptr<ymod_webserver::starter> starter =
        boost::make_shared<ymod_webserver::starter>(
            *io_.get_io(), weak_from_this(), session, session->session_settings());
    starter->logger(logger());
    starter->run();
}

void yplatform_net_server::on_destroy(starter* starter, const process_result& result)
{
    assert(starter);
    switch (result.state)
    {
    case process_result::processing:
        YLOG_CTX_LOCAL(starter->ctx(), error)
            << "preprocessing not finished: "
            << "connection=\"" << starter->get_session()->ctx() << "\"";
        break;
    case process_result::continue_http:
        assert(result.request);
        assert(result.read_buffer);
        execute_http(starter, result.request, result.read_buffer);
        stats_->new_request();
        break;
    case process_result::continue_websocket:
        assert(result.request);
        assert(result.read_buffer);
        execute_websocket(starter, result.request, result.read_buffer);
        stats_->new_request();
        break;
    default:
        break;
    }
}

void yplatform_net_server::on_destroy(http_stream* stream)
{
    assert(stream);
    auto session = stream->get_session();

    stats_->increment_http_code_counters(stream->request()->bound_path, stream->result_code());
    accumulate_http_timing_stats(stream);

    write_2_access_log(settings_, stream);
    if (!settings_->connect_per_request && stream->is_keep_alive())
    {
        session->current_io_service().dispatch(
            std::bind(&yplatform_net_server::on_session_ready, shared_from_this(), session));
    }
}

void yplatform_net_server::on_destroy(net_session* netsession)
{
    assert(netsession);
    session* session = cast_session(netsession);
    if (!session) return;

    accumulate_session_stats(session);
    // TODO implement waiting in net_server

    YLOG_CTX_LOCAL(session->ctx(), info)
        << "session closed: " << session->ctx()->profilers.get_values_with_total()
        << " requests_count=" << session->requests_count();
}

void yplatform_net_server::on_destroy(request* request)
{
    assert(request);
    YLOG_CTX_LOCAL(request->context, info)
        << "request processed: " << request->context->profilers.get_values_with_total();
}

void yplatform_net_server::on_destroy(websocket::websocket_stream* stream)
{
    assert(stream);
    stats_->websocket_streams_count--;
    stats_->increment_websocket_code_counters(stream->request()->bound_path, stream->result_code());
    accumulate_websocket_timing_stats(stream);
    write_2_access_log(settings_, stream);
}

void yplatform_net_server::execute_http(
    starter* starter,
    request_ptr request,
    read_buffer_ptr read_buffer)
{
    assert(starter);
    try
    {
        YLOG_CTX_LOCAL(request->context, debug)
            << "new request: url=\"" << request->raw_request_line << "\" connection=\""
            << starter->get_session()->ctx()->uniq_id() << "\"";

        request->on_destroy_callback = boost::bind(notify_request_destroyed, weak_from_this(), _1);
        session_ptr session = cast_session(starter->get_session());
        if (!session) return;

        boost::shared_ptr<http_stream> http_stream =
            make_http_stream(session, request, read_buffer);

        handler_ptr handler = session->handler();
        if (!handler)
        {
            http_stream->result(codes::not_found);
            return;
        }

        set_request_id(*request);
        if (!set_request_timeout(*request))
        {
            http_stream->result(codes::bad_request, "request timeout too small");
            return;
        }
        try
        {
            handler->execute(request, http_stream);
        }
        catch (const ymod_webserver::http_error& e)
        {
            process_exception(e, http_stream);
            NETSERVER_CTX_LOG(request->context, error)
                << "event=\"http_error exception\" message=\"" << e.public_message() << " "
                << e.private_message() << "\"";
            return;
        }
    }
    catch (const std::exception& e)
    {
        NETSERVER_CTX_LOG(request->context, error)
            << "event=exception message=\"" << e.what() << "\"";
    }

    // DONT FORGET
    // we may polling if connect_per_request because in this case
    // it's not possible to read data from next request
    //  if (settings_.connect_per_request)
    //    begin_poll_connect();
}

void yplatform_net_server::execute_websocket(
    starter* starter,
    request_ptr request,
    read_buffer_ptr read_buffer)
{
    assert(starter);
    try
    {
        YLOG_CTX_LOCAL(request->context, debug)
            << "new websocket request: url=\"" << request->raw_request_line << "\" connection=\""
            << starter->get_session()->ctx()->uniq_id() << "\"";

        request->on_destroy_callback = boost::bind(notify_request_destroyed, weak_from_this(), _1);
        session_ptr session = cast_session(starter->get_session());
        if (!session) return;

        // 404 if no handler
        handler_ptr handler = session->handler();
        if (!handler)
        {
            make_http_stream(session, request, read_buffer)->result(codes::not_found);
            return;
        }

        // notify the handler about upgrading to websocket
        try
        {
            handler->upgrade_to_websocket(request);
        }
        catch (const ymod_webserver::http_error& e)
        {
            process_exception(e, make_http_stream(session, request, read_buffer));
            NETSERVER_CTX_LOG(request->context, error)
                << "event=\"http_error exception\" message=\"" << e.public_message() << " "
                << e.private_message() << "\"";
            return;
        }

        websocket::stream_ptr ws_stream = make_websocket_stream(session, request, read_buffer);

        if (stats_->websocket_streams_count > settings_->ws_max_streams)
        {
            default_answers::send_service_unavailable(ws_stream);
        }
        else
        {
            handler->execute_websocket(ws_stream);
        }
    }
    catch (const std::exception& e)
    {
        NETSERVER_CTX_LOG(request->context, error)
            << "event=exception message=\"" << e.what() << "\"";
    }
}

boost::shared_ptr<http_stream> yplatform_net_server::make_http_stream(
    session_ptr session,
    request_ptr request,
    read_buffer_ptr read_buffer)
{
    boost::shared_ptr<http_stream> stream = boost::make_shared<http_stream>(
        *io_.get_io(), weak_from_this(), request->context, session, request, read_buffer);
    stream->init();
    return stream;
}

boost::shared_ptr<websocket::websocket_stream> yplatform_net_server::make_websocket_stream(
    session_ptr session,
    request_ptr request,
    read_buffer_ptr read_buffer)
{
    const auto& settings = session->session_settings();
    boost::shared_ptr<websocket::websocket_stream> stream =
        boost::make_shared<websocket::websocket_stream>(
            weak_from_this(),
            request->context,
            session,
            request,
            read_buffer,
            logger(),
            settings.ws_max_length,
            settings.ws_max_fragmentation);
    stream->init();
    stats_->websocket_streams_count++;
    return stream;
}

session_ptr yplatform_net_server::cast_session(net_session_ptr netsession)
{
    session_ptr result = boost::dynamic_pointer_cast<ymod_webserver::session>(netsession);
    if (!result)
    {
        NETSERVER_CTX_LOG(netsession->ctx(), error) << "event=\"session cast failed\"";
    }
    return result;
}

session* yplatform_net_server::cast_session(net_session* netsession)
{
    session* result = dynamic_cast<ymod_webserver::session*>(netsession);
    if (!result)
    {
        NETSERVER_CTX_LOG(netsession->ctx(), error) << "event=\"session cast failed\"";
    }
    return result;
}

void yplatform_net_server::set_request_id(request& req)
{
    auto it = req.headers.find("x-request-id");
    if (it == req.headers.end())
        return;
    if (!req.ctx()->request_id().empty()) // This is unexpected.
    {
        YLOG_CTX_LOCAL(req.ctx(), warning)
            << "set_request_id: will overwrite the following value of `request_id`: "
            << req.ctx()->request_id();
    }
    req.ctx()->set_request_id(it->second);
    if (settings_->augment_unique_id_with_request_id)
    {
        req.ctx()->append_to_uniq_id(it->second);
    }
}

bool yplatform_net_server::set_request_timeout(request& req)
{
    auto it = req.headers.find("x-request-timeout");
    if (it != req.headers.end())
    {
        try
        {
            auto timeout = time_traits::milliseconds(std::stoll(it->second));
            if (timeout < settings_->min_acceptable_timeout)
            {
                return false;
            }
            req.ctx()->deadline_from_now(timeout);
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(req.ctx(), warning) << "set_request_timeout exception: " << e.what();
        }
    }
    return true;
}

void yplatform_net_server::accumulate_session_stats(net_session* session)
{
    auto& session_stats = session->stats();
    stats_->session_stats.handshake_errors += session_stats.handshake_errors;
    stats_->session_stats.read_errors += session_stats.read_errors;
    stats_->session_stats.write_errors += session_stats.write_errors;
    stats_->session_stats.ssl_errors += session_stats.ssl_errors;
}

void yplatform_net_server::accumulate_http_timing_stats(http_stream* stream)
{
    auto time = exec_time_in_sec(stream->ctx()->profilers);
    if (std::isnan(time)) return;
    stats_->add_http_timing(stream->request()->bound_path, time);
}

void yplatform_net_server::accumulate_websocket_timing_stats(websocket::websocket_stream* stream)
{
    auto time = exec_time_in_sec(stream->ctx()->profilers);
    if (std::isnan(time)) return;
    stats_->add_websocket_timing(stream->request()->bound_path, time);
}

}
