#include "parser/body.h"
#include "parser/uri.h"
#include "parser/header.h"
#include "parser/char_sets.h"
#include "parser/content_type.h"

namespace ymod_webserver { namespace parser {

namespace { // multipart implementation details

template <typename Iterator, typename BoundaryIter>
class boundary_detector
{
public:
    boundary_detector(BoundaryIter b_begin, BoundaryIter b_end)
        : detected_(false), finally_(false), b_begin_(b_begin), b_end_(b_end)
    {
    }

    bool operator()(Iterator begin, Iterator end)
    {
        while (begin != end)
        {
            for (boundary_start_ = begin; begin != end && !symbols::is_crlf(*begin);
                 boundary_start_ = ++begin)
                ;
            if (begin == end) return false;
            if (!symbols::is_lf(*begin))
            {
                if (++begin == end) return false;
                if (!symbols::is_lf(*begin)) continue;
            }
            if (check_boundary(++begin, end)) return true;
        }
        return false;
    }

    void reset()
    {
        detected_ = false;
        finally_ = false;
    }

    Iterator boundary_start() const
    {
        return boundary_start_;
    }

    Iterator boundary_end() const
    {
        return boundary_end_;
    }

    bool is_detected() const
    {
        return detected_;
    }

    bool is_finally() const
    {
        return finally_;
    }

private:
    bool check_boundary(Iterator begin, Iterator end)
    {
        if (begin == end || *begin != '-') return false;
        ++begin;
        if (begin == end || *begin != '-') return false;
        return check_boundary_equal(++begin, end);
    }

    bool check_boundary_equal(Iterator begin, Iterator end)
    {
        if (begin == end) return false;
        for (BoundaryIter i_bound = b_begin_; i_bound != b_end_; ++i_bound)
        {
            if (*i_bound != *begin) return false;
            if (++begin == end) return false;
        }
        finally_ = (*begin == '-');
        if (finally_)
        {
            ++begin;
            if (begin == end || *begin != '-') return false;
            if (++begin == end) return false;
        }
        if (symbols::is_cr(*begin))
            if (++begin == end) return false;
        if (!symbols::is_lf(*begin)) return false;
        boundary_end_ = ++begin;
        detected_ = true;
        return true;
    }

    Iterator boundary_start_;
    Iterator boundary_end_;
    bool detected_;
    bool finally_; // is boundary is finally
    const BoundaryIter b_begin_;
    const BoundaryIter b_end_;
};

template <typename Iterator>
Iterator parse_part_headers(content_part& part, Iterator begin, Iterator end)
{
    header_parser<Iterator> hparser(&part);
    hparser(begin, begin, end);
    if (!hparser.is_finished()) return end;
    parse_content_type(part);
    return begin;
}

}

void parse_application(request_ptr& req)
{
    parse_params(req->raw_body.begin(), req->raw_body.end(), req->url);
}

void parse_multipart(request_ptr& req)
{
    typedef yplatform::zerocopy::segment::const_iterator iterator_t;
    typedef boundary_detector<iterator_t, string::const_iterator> boundary_detector_t;
    boundary_detector_t detector(req->content.boundary.cbegin(), req->content.boundary.cend());
    auto i_data = req->raw_body.cbegin();
    const auto i_data_end = req->raw_body.cend();
    if (!detector(i_data, i_data_end)) return;
    req->body = req->raw_body.get_part(i_data, detector.boundary_start());
    i_data = detector.boundary_end();
    while (i_data != i_data_end && !detector.is_finally())
    {
        detector.reset();
        if (!detector(i_data, i_data_end)) return;
        req->childs.push_back(content_part());
        const auto body_begin =
            parse_part_headers(req->childs.back(), i_data, detector.boundary_start());
        if (body_begin == detector.boundary_start()) return;
        req->childs.back().body = req->raw_body.get_part(body_begin, detector.boundary_start());
        i_data = detector.boundary_end();
    }
}

void parse_body(request_ptr& req)
{
    if (yplatform::util::iequals(req->content.type, "application") &&
        yplatform::util::iequals(req->content.subtype, "x-www-form-urlencoded"))
    {
        parse_application(req);
    }
    else if (yplatform::util::iequals(req->content.type, "multipart"))
    {
        parse_multipart(req);
    }
}

}}
