#pragma once

#include <functional>
#include <algorithm>
#include <boost/lexical_cast.hpp>
#include <parser/uri.h>
#include <parser/header.h>
#include <parser/body.h>

namespace ymod_webserver { namespace parser {

methods::http_method get_method_by_name(const string& name);
const string& get_method_name(methods::http_method method);

template <typename Iterator>
class request_parser
{
    enum
    {
        policy_request_full_length = (sizeof("<policy-file-request/>") - 1)
    };

public:
    typedef header_parser<Iterator> header_parser_t;
    typedef std::pair<Iterator, bool> check_result;
    typedef Iterator (
        request_parser::*state_handler_t)(Iterator begin, Iterator& start, Iterator end);

    request_parser() : headers_parser_(), policy_read_length_(0)
    {
    }

    Iterator operator()(Iterator begin, Iterator& start, Iterator end)
    {
        state_handler_t prev_handler;
        Iterator detach_point = begin;
        while (true)
        {
            prev_handler = current_handler_;
            detach_point = (this->*(current_handler_))(detach_point, start, end);
            if (prev_handler == current_handler_ || is_finished() || detach_point == end) break;
        }
        return detach_point;
    }

    void reset(context_ptr ctx)
    {
        headers_parser_.reset();
        current_handler_ = &request_parser::on_start;
        current_ = boost::make_shared<request>(std::move(ctx));
        policy_read_length_ = 0;
        request_line_finished_ = false;
    }

    request_ptr& req()
    {
        return current_;
    }

    bool request_line_finished() const
    {
        return request_line_finished_;
    }

    bool is_finished() const
    {
        return (current_->proto_version.first == 0 && current_->proto_version.second == 9) ||
            (headers_parser_.get() && headers_parser_->is_finished()) ||
            (policy_read_length_ == policy_request_full_length);
    }

    bool is_policy_request() const
    {
        return policy_read_length_;
    }

private:
    /**
     * begin, end - range bounds
     * start - iterator in this range to start parse from
     *
     * return value - data between @begin and return value is needless and can be detached/free'd
     */
    Iterator on_start(Iterator begin, Iterator& start, Iterator end);
    Iterator on_policy_request(Iterator begin, Iterator& start, Iterator end);
    Iterator on_method(Iterator begin, Iterator& start, Iterator end);
    Iterator on_uri(Iterator begin, Iterator& start, Iterator end);
    Iterator on_http_version(Iterator begin, Iterator& start, Iterator end);
    Iterator on_request_line_finish(Iterator begin, Iterator& start, Iterator end);
    Iterator on_headers(Iterator begin, Iterator& start, Iterator end);

    static bool is_uri_end_pred(int i)
    {
        return symbols::is_whitespace(static_cast<uint8_t>(i)) ||
            symbols::is_crlf(static_cast<uint8_t>(i));
    }

private: // members
    state_handler_t current_handler_;
    request_ptr current_;
    std::unique_ptr<header_parser_t> headers_parser_;
    unsigned policy_read_length_;
    bool request_line_finished_ = false;
};

template <typename Iterator> // skip empty lines before request
Iterator request_parser<Iterator>::on_start(Iterator, Iterator& start, Iterator end)
{
    while (start != end && symbols::is_crlf(*start))
        ++start;
    if (start == end) return end;
    if (*start == '<') current_handler_ = &request_parser::on_policy_request;
    else
        current_handler_ = &request_parser::on_method;
    return start;
}

template <typename Iterator>
Iterator request_parser<Iterator>::on_policy_request(
    Iterator /*begin*/,
    Iterator& start,
    Iterator end)
{
    const char* policy_str = "<policy-file-request/>";
    for (const char* i_policy = policy_str + policy_read_length_;
         start != end && policy_read_length_ < policy_request_full_length && *start == *i_policy;
         ++start, ++i_policy, ++policy_read_length_)
        ;
    //@todo throw if incorrect
    return start;
}

template <typename Iterator>
Iterator request_parser<Iterator>::on_method(Iterator begin, Iterator& start, Iterator end)
{
    auto method_end = std::find_if(start, end, symbols::is_sp);
    if (method_end == begin)
    {
        throw parse_error("request_parser", "bad request: request line begins with space symbol")
            << BOOST_ERROR_INFO;
    }

    auto crlf = std::find_if_not(start, method_end, symbols::is_alpha);
    if (crlf != method_end)
    {
        throw parse_error("request_parser", "bad request: unexpected symbol") << BOOST_ERROR_INFO;
    }

    start = method_end;
    if (method_end == end) return begin;

    current_->method = get_method_by_name(string(begin, method_end));
    if (current_->method == methods::mth_invalid)
    {
        throw parse_error("request_parser", "invalid method: " + string(begin, method_end))
            << BOOST_ERROR_INFO;
    }
    current_->raw_request_line.append(begin, method_end);
    current_->raw_request_line.append(" ");
    current_handler_ = &request_parser::on_uri;
    return (start = method_end);
}

template <typename Iterator>
Iterator request_parser<Iterator>::on_uri(Iterator /*begin*/, Iterator& start, Iterator end)
{
    // skip whitespaces
    auto uri_start = std::find_if_not(start, end, symbols::is_sp);
    if (uri_start == end) return (start = uri_start);

    auto uri_end = std::find_if(uri_start, end, is_uri_end_pred);
    if (uri_end == end) return (start = uri_start);

    if (!parse_uri<Iterator>(uri_start, uri_end, current_->url))
    {
        throw parse_error("request_parser", "invalid uri: " + string(uri_start, uri_end))
            << BOOST_ERROR_INFO;
    }
    current_handler_ = &request_parser::on_http_version;
    current_->raw_url.append(uri_start, uri_end);
    current_->raw_request_line.append(uri_start, uri_end);
    current_->raw_request_line.append(" ");
    return (start = uri_end);
}

#define COMPARE_AND_INCREMENT(i, sym)                                                              \
    if (i == version_end || *i != sym)                                                             \
    {                                                                                              \
        throw parse_error("request_parser", "invalid http_version: " + string(start, version_end)) \
            << BOOST_ERROR_INFO;                                                                   \
    }                                                                                              \
    else                                                                                           \
    {                                                                                              \
        ++i;                                                                                       \
    }

template <typename Iterator>
Iterator request_parser<Iterator>::on_http_version(
    Iterator /*begin*/,
    Iterator& start,
    Iterator end)
{
    // skip whitespaces
    start = std::find_if_not(start, end, symbols::is_sp);
    if (start == end) return start;

    // check if it is Simple-Request or Full-Request
    if (symbols::is_crlf(*start))
    {
        current_->proto_version = std::make_pair(0, 9);
        current_handler_ = &request_parser::on_request_line_finish;
        return start;
    }

    auto version_end = start;
    while (version_end != end && !symbols::is_crlf(*version_end))
        ++version_end;
    if (version_end == end) return start;

    auto i_ver = start;
    COMPARE_AND_INCREMENT(i_ver, 'H');
    COMPARE_AND_INCREMENT(i_ver, 'T');
    COMPARE_AND_INCREMENT(i_ver, 'T');
    COMPARE_AND_INCREMENT(i_ver, 'P');
    COMPARE_AND_INCREMENT(i_ver, '/');
    for (; i_ver != version_end && std::isdigit(*i_ver); ++i_ver)
    {
        current_->proto_version.first =
            static_cast<uint8_t>(10 * current_->proto_version.first + (*i_ver - '0'));
    }
    COMPARE_AND_INCREMENT(i_ver, '.');
    for (; i_ver != version_end && std::isdigit(*i_ver); ++i_ver)
    {
        current_->proto_version.second =
            static_cast<uint8_t>(10 * current_->proto_version.second + (*i_ver - '0'));
    }
    current_->raw_request_line.append(start, version_end);
    current_handler_ = &request_parser::on_request_line_finish;
    headers_parser_.reset(new header_parser_t(current_.get()));
    return (start = version_end);
}
#undef COMPARE_AND_INCREMENT

template <typename Iterator>
Iterator request_parser<Iterator>::on_request_line_finish(
    Iterator /*begin*/,
    Iterator& start,
    Iterator end)
{
    request_line_finished_ = true;
    while (start != end && symbols::is_cr(*start))
        ++start;
    if (start == end) return start;
    if (symbols::is_lf(*start))
    {
        if (current_->proto_version.first > 0)
        {
            current_handler_ = &request_parser::on_headers;
        }
        return ++start;
    }
    throw parse_error("request_parser", "invalid http/0.9 request") << BOOST_ERROR_INFO;
}

template <typename Iterator>
Iterator request_parser<Iterator>::on_headers(Iterator begin, Iterator& start, Iterator end)
{
    return (*headers_parser_)(begin, start, end);
}

}}
