#pragma once

#include "parser/common.h"
#include "parser/char_sets.h"
#include <ymod_webserver/uri.h>
#include <ymod_webserver/error.h>

namespace ymod_webserver { namespace parser {

template <typename Iterator>
Iterator parse_params(Iterator begin, Iterator end, http_uri& u)
{
    Iterator param_name = begin;
    Iterator start = begin;
    while (start != end && *start != '#')
    {
        for (; start != end && *start != '&' && *start != ';' && *start != '=' && *start != '#';
             ++start)
            ;
        param_name = start;
        if (start != end && *start != '#' && *start == '=')
        {
            for (; start != end && *start != '&' && *start != ';' && *start != '#'; ++start)
                ;
            Iterator param_value = param_name;
            u.params.insert(std::make_pair(
                encode_param_range(begin, param_name), encode_param_range(++param_value, start)));
        }
        else
        {
            u.params.insert(std::make_pair(encode_param_range(begin, param_name), ""));
        }
        if (start != end && *start != '#')
        {
            begin = ++start;
        }
    }
    return start;
}

namespace {

template <typename Iterator>
Iterator parse_port(Iterator begin, Iterator end, host_info& host)
{
    host.port = 0;
    while (begin != end && std::isdigit(*begin))
    {
        host.port = 10 * host.port + *begin - '0';
        ++begin;
    }
    return begin;
}

template <typename Iterator>
Iterator skip_proto_slashes(Iterator& start, Iterator end)
{
    if (++start == end || *start != '/') throw parse_error() << BOOST_ERROR_INFO;
    if (start == end) throw parse_error() << BOOST_ERROR_INFO;
    return ++start; // skip '/'
}

} // namespace

template <typename Iterator>
Iterator parse_host(Iterator begin, Iterator end, host_info& host)
{
    Iterator start = begin;
    for (; start != end && symbols::is_alpha(*start); ++start)
        ;
    if (start == end)
    {
        host.domain.assign(begin, start);
        return start;
    }
    // if <proto>:// or <domain>:<port>
    if (*start == ':')
    {
        Iterator it_colon = start++;
        if (start == end) throw parse_error("ending colon") << BOOST_ERROR_INFO;
        // if proto
        if (*start == '/')
        {
            host.proto.assign(begin, it_colon);
            begin = skip_proto_slashes(start, end);
            // if host+port
        }
        else if (symbols::is_digit(*start))
        {
            host.domain.assign(begin, it_colon);
            return parse_port(start, end, host);
            // else bad request
        }
        else
        {
            throw parse_error() << BOOST_ERROR_INFO;
        }
    }

    // parse host[:port] (find ':' or '/')
    while (start != end && *start != ':' && *start != '/')
        ++start;
    host.domain.assign(begin, start);
    if (start != end && *start == ':')
    {
        if (++start != end && symbols::is_digit(*start))
        {
            host.port = 0;
            start = parse_port(start, end, host);
        }
    }
    return start;
}

template <typename Iterator>
bool parse_uri(Iterator begin, Iterator end, http_uri& uri)
{
    if (begin == end) return false;
    if (*begin != '/') begin = parse_host(begin, end, uri.host);
    if (begin == end || *begin != '/') return begin == end;
    Iterator start = ++begin;
    while (start != end && *start != '?' && *start != '#')
    {
        while (start != end && *start != '/' && *start != '?' && *start != '#')
            ++start;
        if (begin != start) uri.path.push_back(encode_param_range(begin, start));
        if (start != end && *start == '/') begin = ++start;
    }
    if (start == end) return true;
    if (*start == '?')
    {
        ++start;
        start = parse_params(start, end, uri);
    }
    if (start == end) return true;
    if (*start == '#')
    {
        ++start;
        uri.fragment.assign(start, end);
    }
    return true;
}

}}
