#include "protect_cookies.h"
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/range/iterator_range.hpp>
#include <string>
#include <algorithm>

using string_range = boost::iterator_range<std::string::iterator>;

static bool parse(
    const string_range& cookie_pair,
    string_range& out_cookie_name,
    string_range& out_cookie_value);
static bool need_to_protect(const string_range& cookie_name);
static void protect(const string_range& cookie_value);

namespace ymod_webserver {

void protect_sensitive_cookies(std::string& cookies)
{
    using namespace boost;

    // RFC 6265: cookie-string = cookie-pair *( ";" SP cookie-pair )
    // For the purpose of better security, drop SP from cookie delimiter.
    // Thus we protect sensitive cookies, even if they are not strictly RFC-compliant.
    // Worst case scenario - we overwrite more stuff than we should.
    static const std::string cookie_pair_delimiter = ";";

    auto split_begin =
        make_split_iterator(cookies, first_finder(cookie_pair_delimiter, is_iequal()));
    auto split_end = split_iterator<std::string::iterator>();

    string_range cookie_name;
    string_range cookie_value;

    for (auto cookie_pair = split_begin; cookie_pair != split_end; ++cookie_pair)
    {
        if (parse(*cookie_pair, cookie_name, cookie_value) && need_to_protect(cookie_name))
        {
            protect(cookie_value);
        }
    }
}

}

bool parse(const string_range& cookie_pair, string_range& cookie_name, string_range& cookie_value)
{
    // RFC 6265: cookie-pair = cookie-name "=" cookie-value
    cookie_name = boost::make_iterator_range(
        cookie_pair.begin(), std::find(cookie_pair.begin(), cookie_pair.end(), '='));
    if (cookie_name.end() == cookie_pair.end())
    {
        return false;
    }

    cookie_value = boost::make_iterator_range(std::next(cookie_name.end()), cookie_pair.end());
    return cookie_value.begin() != cookie_pair.end();
}

bool need_to_protect(const string_range& cookie_name)
{
    static const std::string protectd_cookies_lowercase[] = {
        "session_id", "sessionid2", "secure_session_id", "golem_session"
    };

    for (const auto& name_to_check : protectd_cookies_lowercase)
    {
        if (boost::algorithm::icontains(cookie_name, name_to_check))
        {
            return true;
        }
    }

    return false;
}

void protect(const string_range& cookie_value)
{
    static const std::string replacement_message = "_|^|";

    std::string::reverse_iterator cookie_value_rbegin(cookie_value.end());
    std::string::reverse_iterator cookie_value_rend(cookie_value.begin());

    auto protect_rend = std::find(cookie_value_rbegin, cookie_value_rend, '.');
    if (protect_rend != cookie_value_rend)
    {
        auto replacement_symbol = replacement_message.rbegin();
        std::generate(cookie_value_rbegin, protect_rend, [&replacement_symbol]() -> char {
            if (replacement_symbol == replacement_message.rend())
            {
                replacement_symbol = replacement_message.rbegin();
            }
            return *replacement_symbol++;
        });
    }
}
