#include "session.h"

#include <boost/asio/ssl/error.hpp>
#include <boost/bind/protect.hpp>
#include <yplatform/encoding/url_encode.h>
#include "net_server.h"

namespace ymod_webserver {

bool connection_is_closed(boost::system::error_code const& e)
{
    // 'short read' error happens when work over ssl
    // it's not clear case so we just skip it
    static auto short_read = boost::asio::ssl::error::stream_truncated;
    return e == boost::asio::error::eof || e == short_read || e == boost::asio::error::shut_down ||
        e == boost::asio::error::broken_pipe || e == boost::asio::error::connection_reset;
}

session::session(
    yplatform::net::tcp_socket&& socket,
    const ymod_webserver::settings& settings,
    ymod_webserver::endpoint& endpoint,
    handler_ptr handler)
    : socket_(std::move(socket))
    , settings_(settings)
    , endpoint_(endpoint)
    , write_active_(false)
    , handler_(handler)
{
    std::stringstream stream;
    stream << "ywebserver_session " << id();
    logger().set_log_prefix(logger().get_log_prefix() + stream.str());
    make_context();
}

session::~session()
{
    if (net_server_ptr powner = owner_.lock()) try
        {
            powner->on_destroy(this);
        }
        catch (...)
        {
        }
}

void session::make_context()
{
    ctx_.reset(new context);
    ctx_->remote_address = remote_addr().to_string();
    ctx_->remote_port = remote_port();
    ctx_->local_address = local_addr().to_string();
    ctx_->local_port = local_port();
}

void session::real_enable_ssl(enable_ssl_hook_t const& hook)
{
    ctx_->state = context_state_t::handshake;
    secure_ = true;
    socket_.async_tls_handshake(
        yplatform::net::tcp_socket::handshake_type::server,
        endpoint_.socket_settings.tls_timeout,
        hook);
}

void session::handle_handshake(const boost::system::error_code& e, enable_ssl_hook_t const& hook)
{
    if (e) update_stats_on_handshake_error(e);
    hook(e);
}

void session::real_begin_read(
    read_hook_t const& hook,
    mutable_read_buffer_t buffer,
    std::size_t min)
{
    if (!this->is_open())
    {
        hook(boost::asio::error::eof, 0);
        return;
    }

    socket_.async_read(
        buffer,
        endpoint_.socket_settings.read_timeout,
        boost::protect(boost::bind(&session::handle_read, this->shared_from_this(), _1, _2, hook)),
        min);
}

void session::handle_read(
    const boost::system::error_code& e,
    std::size_t bytes,
    read_hook_t const& hook)
{
    if (e) update_stats_on_read_error(e);
    ctx_->bytes_received += bytes;
    hook(e, bytes);
}

void session::real_update_read_timeout(yplatform::time_traits::duration const& value)
{
    endpoint_.socket_settings.read_timeout = value;
}

bool session::begin_write()
{
    try
    {
        if (!writeq_.flush())
        {
            socket_.async_write(
                writeq_.send_queue(),
                endpoint_.socket_settings.write_timeout,
                boost::protect(
                    boost::bind(&session::handle_write, this->shared_from_this(), _1, _2)));
            write_active_ = true;
            return true;
        }
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "begin_write exception message=\"" << e.what() << "\"";
        update_stats_on_write_error();
        if (write_error_hook_) write_error_hook_(boost::asio::error::fault);
    }
    catch (...)
    {
        YLOG_L(error) << "begin_write exception message=none";
        update_stats_on_write_error();
        if (write_error_hook_) write_error_hook_(boost::asio::error::fault);
    }
    return false;
}

void session::handle_write(const boost::system::error_code& e, std::size_t bytes)
{
    ctx_->bytes_sent += bytes;
    if (e)
    {
        update_stats_on_write_error(e);
        socket_.close();
        if (write_error_hook_) write_error_hook_(e);
        return;
    }
    writeq_.consume(bytes);
    write_active_ = false;
    if (!begin_write())
    {
        // TODO on_write_finished(); ?
    }
}

yplatform::net::streamer_wrapper session::client_stream()
{
    return yplatform::net::streamer_wrapper(
        new yplatform::net::streamer<session>(shared_from_this()));
}

void session::send_client_stream(yplatform::net::buffers::const_chunk_buffer const& s)
{
    socket_.get_io()->dispatch(
        boost::bind(&session::real_send_client_stream, shared_from_this(), s));
}

void session::send_client_stream2(
    yplatform::net::buffers::const_chunk_buffer const& s,
    bool /*skip_log*/)
{
    send_client_stream(s);
}

void session::real_send_client_stream(yplatform::net::buffers::const_chunk_buffer const& s)
{
    if (!this->is_open()) return;

    try
    {
        writeq_.push(s);
    }
    catch (std::exception& e)
    {
        YLOG_L(info) << "real_send_client_stream exception message=\"" << e.what() << "\"";
        return;
    }
    catch (...)
    {
        YLOG_L(info) << "real_send_client_stream exception message=none";
        return;
    }

    if (!write_active_) begin_write();
}

void session::update_stats_on_handshake_error(const boost::system::error_code& e)
{
    stats_.handshake_errors += 1;
    if (e.category() == boost::asio::error::get_ssl_category()) stats_.ssl_errors += 1;
}

void session::update_stats_on_read_error(const boost::system::error_code& e)
{
    stats_.read_errors += 1;
    if (e.category() == boost::asio::error::get_ssl_category()) stats_.ssl_errors += 1;
}

void session::update_stats_on_write_error(const boost::system::error_code& e)
{
    stats_.write_errors += 1;
    if (e.category() == boost::asio::error::get_ssl_category()) stats_.ssl_errors += 1;
}

}
