#pragma once

#include <boost/bind/protect.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/locks.hpp>
#include <yplatform/net/line_filter.h>
#include <yplatform/net/socket.h>
#include <yplatform/net/tcp_stats.h>
#include <yplatform/zerocopy/streambuf.h>
#include <yplatform/net/handlers/timer_handler.h>

#include <ymod_webserver/context.h>
#include <ymod_webserver/settings.h>
#include "net_session.h"

namespace ymod_webserver {

class net_server;

bool connection_is_closed(boost::system::error_code const& ec);

class session
    : public net_session
    , public boost::enable_shared_from_this<session>
    , public yplatform::log::contains_logger
{
    typedef yplatform::net::output_buffer_sequence<yplatform::net::buffers::const_chunk_buffer>
        output_queue_t;

public:
    typedef yplatform::zerocopy::streambuf buffer_t;
    typedef boost::function<void(boost::system::error_code const&)> enable_ssl_hook_t;
    typedef boost::function<void(boost::system::error_code const&, std::size_t)> read_hook_t;
    typedef boost::function<void(boost::system::error_code const&)> write_error_hook_t;

    session(
        yplatform::net::tcp_socket&& socket,
        const ymod_webserver::settings& settings,
        ymod_webserver::endpoint& endpoint,
        handler_ptr handler);
    ~session();

    void send_client_stream(yplatform::net::buffers::const_chunk_buffer const&);
    void send_client_stream2(yplatform::net::buffers::const_chunk_buffer const&, bool);
    yplatform::net::streamer_wrapper client_stream();

    // thread unsafe
    // Call after opening the session and strongly before starting all operations.
    // A pointer to an owner is not passed to session's constructor because of
    // session is created inside yplatform::server.
    void set_owner(boost::weak_ptr<net_server> owner)
    {
        owner_ = owner;
    }

    yplatform::task_context_ptr get_context() const
    {
        return ctx_;
    }

    context_ptr ctx()
    {
        return ctx_;
    }

    const settings& session_settings() const
    {
        return settings_;
    }

    boost::asio::io_service& current_io_service()
    {
        return *socket_.get_io();
    }

    const boost::asio::ip::address& remote_addr() const
    {
        return socket_.remote_addr();
    }

    unsigned short remote_port() const
    {
        return socket_.remote_port();
    }

    const boost::asio::ip::address& local_addr() const
    {
        return socket_.local_addr();
    }

    unsigned short local_port() const
    {
        return socket_.local_port();
    }

    void enable_ssl(enable_ssl_hook_t const& hook)
    {
        socket_.get_io()->dispatch(
            boost::bind(&session::real_enable_ssl, shared_from_this(), hook));
    }

    void const* id() const
    {
        return socket_.id();
    }

    bool is_open() const
    {
        return socket_.is_open();
    }

    bool is_secure() const
    {
        return secure_;
    }

    void begin_read(read_hook_t const& hook, mutable_read_buffer_t buffer, std::size_t min)
    {
        socket_.get_io()->dispatch(
            boost::bind(&session::real_begin_read, shared_from_this(), hook, buffer, min));
    }

    void update_read_timeout(yplatform::time_traits::duration const& value)
    {
        socket_.get_io()->dispatch(
            boost::bind(&session::real_update_read_timeout, shared_from_this(), value));
    }

    void async_close(const close_hook_t& handler)
    {
        socket_.get_io()->dispatch([this, self = shared_from_this(), handler]() {
            collect_stats();
            socket_.close();
            handler(boost::system::error_code());
        });
    }

    void close()
    {
        collect_stats();
        socket_.close();
    }

    void do_shutdown(const shutdown_hook_t& handler, bool graceful)
    {
        socket_.get_io()->dispatch([this, self = shared_from_this(), graceful, handler]() {
            collect_stats();
            socket_.shutdown(graceful);
            handler();
        });
    }

    void cancel_operations(const cancel_hook_t& handler)
    {
        socket_.get_io()->dispatch([this, self = shared_from_this(), handler]() {
            socket_.cancel_operations();
            handler();
        });
    }

    void set_write_error_hook(write_error_hook_t const& write_error_hook)
    {
        socket_.get_io()->dispatch(
            boost::bind(&session::real_set_write_error_hook, shared_from_this(), write_error_hook));
    }

    unsigned requests_count() const
    {
        return requests_count_;
    }

    void increment_requests_count()
    {
        requests_count_++;
    }

    const ymod_webserver::endpoint& endpoint() const
    {
        return endpoint_;
    }

    const handler_ptr& handler() const
    {
        return handler_;
    }

    const session_stats& stats()
    {
        collect_stats();
        return stats_;
    }

private:
    void real_enable_ssl(enable_ssl_hook_t const& hook);

    void real_begin_read(read_hook_t const& hook, mutable_read_buffer_t buffer, std::size_t min);

    void real_update_read_timeout(yplatform::time_traits::duration const& value);

    void real_set_write_error_hook(write_error_hook_t const& hook)
    {
        write_error_hook_ = hook;
    }

    void handle_handshake(const boost::system::error_code& e, enable_ssl_hook_t const& hook);

    void handle_read(
        const boost::system::error_code& e,
        std::size_t bytes,
        read_hook_t const& hook);

    void real_send_client_stream(yplatform::net::buffers::const_chunk_buffer const&);

    void make_context();
    bool begin_write();
    void handle_write(boost::system::error_code const& e, std::size_t bytes);

    void collect_stats()
    {
        if (stats_.socket_info.empty())
            stats_.socket_info = yplatform::net::collect_tcp_stats(socket_);
    }

    void update_stats_on_handshake_error(const boost::system::error_code& e = {});
    void update_stats_on_read_error(const boost::system::error_code& e = {});
    void update_stats_on_write_error(const boost::system::error_code& e = {});

    yplatform::net::tcp_socket socket_;
    settings settings_;
    ymod_webserver::endpoint& endpoint_;
    bool write_active_;
    context_ptr ctx_;
    output_queue_t writeq_;
    boost::weak_ptr<net_server> owner_;
    write_error_hook_t write_error_hook_;
    unsigned requests_count_ = 0;
    bool secure_ = false;
    handler_ptr handler_;
    session_stats stats_;
};

typedef boost::shared_ptr<session> session_ptr;
}
