#pragma once

#include <ymod_webserver/error.h>
#include <ymod_webserver/request.h>
#include "parser/char_sets.h"
#include "parser/header.h"

namespace ymod_webserver { namespace transfer {

template <typename Iterator, typename Stream>
class chunked
{
private:
    unsigned hex_value(uint8_t i)
    {
        if (symbols::is_digit(i)) return i - '0';
        if (i >= 'A' && i <= 'F') return 10 + i - 'A';
        if (i >= 'a' && i <= 'f') return 10 + i - 'a';
        throw parse_error() << BOOST_ERROR_INFO;
    }

    typedef Iterator (
        chunked::*state_handler_t)(Iterator begin, Iterator& start, Iterator end, Stream& output);

    typedef parser::header_parser<Iterator> header_parser_t;

public:
    chunked(const request_ptr& request)
        : handler_(&chunked::parse_chunk_prepare)
        , current_size_(0)
        , request_(request)
        , header_(request.get())
        , finished_(false)
    {
    }

    Iterator operator()(Iterator begin, Iterator& start, Iterator end, Stream& output)
    {
        state_handler_t prev_handler;
        while (true)
        {
            prev_handler = handler_;
            begin = (this->*(handler_))(begin, start, end, output);
            if (prev_handler == handler_ || finished_) break;
            start = begin;
        }
        return begin;
    }

    bool is_finished() const
    {
        return finished_;
    }

private:
    Iterator parse_chunk_prepare(Iterator begin, Iterator& start, Iterator end, Stream& output);

    Iterator parse_chunk_size(Iterator begin, Iterator& start, Iterator end, Stream& output);

    Iterator parse_chunk_ext(Iterator begin, Iterator& start, Iterator end, Stream& output);

    Iterator parse_chunk_body(Iterator begin, Iterator& start, Iterator end, Stream& output);

    Iterator parse_chunk_fin(Iterator begin, Iterator& start, Iterator end, Stream& output);

    Iterator parse_chunk_lws(Iterator begin, Iterator& start, Iterator end, Stream& output);

    state_handler_t handler_;
    std::size_t current_size_;
    request_ptr request_;
    header_parser_t header_;
    bool finished_;
};

template <typename Iterator, typename Stream>
Iterator chunked<Iterator, Stream>::parse_chunk_prepare(
    Iterator begin,
    Iterator& start,
    Iterator end,
    Stream& output __attribute__((unused)))
{
    if (start == end) return begin;
    if (!symbols::is_hex(*start)) throw std::runtime_error("invalid chunk size");
    handler_ = &chunked::parse_chunk_size;
    return start;
}

template <typename Iterator, typename Stream>
Iterator chunked<Iterator, Stream>::parse_chunk_size(
    Iterator begin __attribute__((unused)),
    Iterator& start,
    Iterator end,
    Stream& output __attribute__((unused)))
{
    for (; start != end && symbols::is_hex(*start); ++start)
        current_size_ = 16 * current_size_ + hex_value(*start);
    if (start != end) handler_ = &chunked::parse_chunk_ext;
    return start;
}

template <typename Iterator, typename Stream>
Iterator chunked<Iterator, Stream>::parse_chunk_ext(
    Iterator begin __attribute__((unused)),
    Iterator& start,
    Iterator end,
    Stream& output __attribute__((unused)))
{
    for (; start != end && !symbols::is_lf(*start); ++start)
        ;
    if (start != end)
    {
        ++start;
        if (current_size_ == 0) handler_ = &chunked::parse_chunk_fin;
        else
            handler_ = &chunked::parse_chunk_body;
    }
    return start;
}

template <typename Iterator, typename Stream>
Iterator chunked<Iterator, Stream>::parse_chunk_body(
    Iterator begin,
    Iterator& start,
    Iterator end,
    Stream& output)
{
    for (; start != end && current_size_ > 0; ++start, --current_size_)
        ;
    if (start == end) return begin;
    std::copy(
        begin,
        start,
        std::ostream_iterator<typename std::iterator_traits<Iterator>::value_type>(output));
    handler_ = &chunked::parse_chunk_lws;
    return start;
}

template <typename Iterator, typename Stream>
Iterator chunked<Iterator, Stream>::parse_chunk_lws(
    Iterator begin __attribute__((unused)),
    Iterator& start,
    Iterator end,
    Stream& output __attribute__((unused)))
{
    for (; start != end && symbols::is_cr(*start); ++start)
        ;
    if (start == end) return start;
    if (!symbols::is_lf(*start)) throw std::runtime_error("invalid chunk finish");
    for (; start != end && symbols::is_crlf(*start); ++start)
        ;

    current_size_ = 0;
    handler_ = &chunked::parse_chunk_size;
    return start;
}

template <typename Iterator, typename Stream>
Iterator chunked<Iterator, Stream>::parse_chunk_fin(
    Iterator begin,
    Iterator& start,
    Iterator end,
    Stream& output __attribute__((unused)))
{
    Iterator res = header_(begin, start, end);
    finished_ = header_.is_finished();
    return res;
}

}}
