#pragma once

#include <ymod_webserver/request.h>
#include "transfer/compress.h"
#include "transfer/chunked.h"
#include "transfer/deflate.h"
#include "transfer/gzip.h"

namespace ymod_webserver { namespace transfer {

template <typename Buffer, typename Iterator>
class parser
{
    typedef Buffer buffer_t;
    typedef Iterator iterator_t;
    typedef chunked<iterator_t, std::ostream> chunked_parser;
    typedef typename boost::shared_ptr<buffer_t> buffer_ptr;
    typedef iterator_t (parser::*mode_handler_t)(
        iterator_t begin,
        iterator_t& start,
        iterator_t end,
        std::size_t bytes);

public:
    typedef typename buffer_t::segment_type segment_type;

    parser(const request_ptr& req, buffer_ptr input);

    iterator_t operator()(iterator_t begin, iterator_t& start, iterator_t end, std::size_t bytes)
    {
        return (this->*parse_)(begin, start, end, bytes);
    }

    bool is_finished() const
    {
        return finished_;
    }

    bool result_empty() const
    {
        return result_->size() > 0;
    }

    segment_type result(iterator_t& i_saved)
    {
        if (result_ == input_)
        {
            return result_->detach(i_saved);
        }
        else
        {
            return result_->detach(result_->end());
        }
    }

protected:
    iterator_t do_nothing(iterator_t begin, iterator_t& start, iterator_t end, std::size_t bytes);

    iterator_t do_chunked(iterator_t begin, iterator_t& start, iterator_t end, std::size_t bytes);

    bool finished_;
    request_ptr req_;
    std::size_t content_length_;
    buffer_ptr input_;
    buffer_ptr result_;
    mode_handler_t parse_;
    std::unique_ptr<chunked_parser> chunked_;
};

template <typename Buffer, typename Iterator>
parser<Buffer, Iterator>::parser(const request_ptr& req, buffer_ptr input)
    : finished_(false), content_length_(0), parse_(0), chunked_(nullptr)
{
    req_ = req;
    input_ = input;
    content_length_ = req_->content.length;
    if (req->transfer_encoding == transfer_encoding_chunked)
    {
        parse_ = &parser::do_chunked;
        chunked_.reset(new chunked_parser(req));
        result_.reset(new buffer_t);
    }
    else if (req->transfer_encoding == transfer_encoding_identity)
    {
        parse_ = &parser::do_nothing;
        result_ = input_;
    }
    else
    {
        throw std::runtime_error("unsupported transfer encoding");
    }
}

template <typename Buffer, typename Iterator>
typename parser<Buffer, Iterator>::iterator_t parser<Buffer, Iterator>::do_nothing(
    iterator_t begin,
    iterator_t& start,
    iterator_t end,
    std::size_t bytes)
{
    assert(end >= start);
    assert(std::size_t(end - start) >= bytes);
    if (content_length_ <= bytes)
    {
        finished_ = true;
        for (std::size_t c = 0; c < content_length_; ++c, ++start)
            ;
        content_length_ = 0;
    }
    else
    {
        content_length_ -= bytes;
        start = end;
    }
    return begin;
}

template <typename Buffer, typename Iterator>
typename parser<Buffer, Iterator>::iterator_t parser<Buffer, Iterator>::do_chunked(
    iterator_t begin,
    iterator_t& start,
    iterator_t end,
    std::size_t bytes __attribute__((unused)))
{
    std::ostream stream(result_.get());
    iterator_t res = (*chunked_)(begin, start, end, stream);
    finished_ = chunked_->is_finished();
    return res;
}

}}
