#include "validator.h"
#include "parser/uri.h"
#include "parser/util.h"
#include "parser/header_qvalue.h"
#include "parser/content_type.h"
#include <boost/bind.hpp>
#include <boost/assign/list_of.hpp>
#include <yplatform/util.h>
#include <ymod_webserver/error.h>
#include <yplatform/util.h>

namespace ymod_webserver {

namespace {

class transfer_encoding_handler
{
    typedef string::const_iterator iter_t;

public:
    transfer_encoding_handler(request_ptr& req) : req_(req)
    {
    }

    void on_value(iter_t begin, iter_t end)
    {
        if (yplatform::util::iequals(boost::iterator_range<iter_t>(begin, end), "chunked"))
        {
            req_->transfer_encoding = transfer_encoding_chunked;
        }
        else
        {
            req_->transfer_encoding = transfer_encoding_ext;
        }
    }

    void on_param(
        iter_t nbegin __attribute__((unused)),
        iter_t nend __attribute__((unused)),
        iter_t vbegin __attribute__((unused)),
        iter_t vend __attribute__((unused)))
    {
    }

    void finished()
    {
    }

private:
    request_ptr& req_;
};

class connection_handler
{
    typedef string::const_iterator iter_t;

public:
    connection_handler(request_ptr& req) : req_(req)
    {
    }

    void on_value(iter_t begin, iter_t end)
    {
        connection_header v = parser::parse_connection(begin, end);
        if (v != connection_te) req_->connection = v;
    }

    void on_param(
        iter_t nbegin __attribute__((unused)),
        iter_t nend __attribute__((unused)),
        iter_t vbegin __attribute__((unused)),
        iter_t vend __attribute__((unused)))
    {
    }

    void finished()
    {
    }

private:
    request_ptr& req_;
};

}

const validator::header_handlers_t validator::header_handlers_ = boost::assign::map_list_of(
    "connection",
    &validator::on_connection_header)("transfer-encoding", &validator::on_transfer_encoding_header)(
    "upgrade",
    &validator::on_upgrade_header)("host", &validator::on_host_header)(
    "content-length",
    &validator::on_content_length_header)("content-type", &validator::on_content_type_header)(
    "content-encoding",
    &validator::on_content_encoding_header)("sec-websocket-key1", &validator::on_websocket76_key)(
    "sec-websocket-key2",
    &validator::on_websocket76_key)("sec-websocket-key", &validator::on_websocket_key)(
    "sec-websocket-version",
    &validator::on_websocket_version)("origin", &validator::on_origin_header)(
    "accept",
    &validator::on_accept_header)("accept-encoding", &validator::on_accept_encoding_header);

void validator::on_connection_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    connection_handler handler(req);
    parser::parse_header_value(i_header->second.begin(), i_header->second.end(), handler);
}

void validator::on_transfer_encoding_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    transfer_encoding_handler handler(req);
    parser::parse_header_value(i_header->second.begin(), i_header->second.end(), handler);
}

void validator::on_upgrade_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    if (req->upgrade_to != upgrade_none) return;
    if (yplatform::util::iequals(i_header->second, "websocket"))
    {
        if (req->headers.find("sec-websocket-key1") == req->headers.end())
            req->upgrade_to = upgrade_to_websocket75;
        else
            req->upgrade_to = upgrade_to_websocket76;
    }
    else
    {
        req->upgrade_to = upgrade_to_unknown_proto;
    }
}

void validator::on_host_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    vhost_parsed_ = true;
    req->vhost.port = 0;
    parser::parse_host(i_header->second.begin(), i_header->second.end(), req->vhost);
}

void validator::on_content_length_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    static_assert(
        sizeof(long long) == sizeof(std::size_t), "long long cant be casted safely to std::size_t");

    long long raw_length;
    try
    {
        raw_length = boost::lexical_cast<long long>(i_header->second);
    }
    catch (const boost::bad_lexical_cast& ex)
    {
        throw parse_error("validator", "bad lexical cast of content length", "")
            << BOOST_ERROR_INFO;
    }

    if (raw_length < 0)
    {
        throw parse_error("validator", "content length < 0", "") << BOOST_ERROR_INFO;
    }

    req->content.length = raw_length;
}

void validator::on_content_type_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    parser::content_type_handler handler(req->content);
    parser::parse_header_value(i_header->second.begin(), i_header->second.end(), handler);
}

void validator::on_content_encoding_header(request_ptr& req, header_map_t::const_iterator iter)
{
    req->content_encoding =
        parser::parse_content_encoding(iter->second.begin(), iter->second.end());
}

void validator::on_websocket76_key(request_ptr&, header_map_t::const_iterator iter)
{
    if (iter->second.find(' ') == iter->second.npos)
        throw parse_error(
            "validator",
            "incorrect websocket76 key header (space require)",
            "incorrect websocket76 key header (space require)")
            << BOOST_ERROR_INFO;

    ++websocket_count_;
}

void validator::on_websocket_key(request_ptr&, header_map_t::const_iterator)
{
    ++websocket_count_;
}

void validator::on_websocket_version(request_ptr& req, header_map_t::const_iterator iter)
{
    int proto_ver = atoi(iter->second.c_str());
    switch (proto_ver)
    {
    case 7:
        req->upgrade_to = upgrade_to_websocket07;
        break;
    case 8:
        req->upgrade_to = upgrade_to_websocket08;
        break;
    case 13:
        req->upgrade_to = upgrade_to_websocket13;
        break;
    default:
        req->upgrade_to = upgrade_to_websocket_unknown;
        break;
    }
}

void validator::on_origin_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    req->origin = host_info();
    parser::parse_host(i_header->second.begin(), i_header->second.end(), req->origin.get());
}

void validator::on_accept_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    req->accept = accept_header_value_type();
    parser::parse_header_value_copy_handler(
        i_header->second.begin(),
        i_header->second.end(),
        parser::make_header_qvalue<string::const_iterator>(
            req->accept.get(),
            boost::bind(&parser::parse_content_type_value<string::const_iterator>, _1, _2, _3)));
}

void validator::on_accept_encoding_header(request_ptr& req, header_map_t::const_iterator i_header)
{
    req->accept_encoding = accept_encoding_header_value_type();
    parser::parse_header_value_copy_handler(
        i_header->second.begin(),
        i_header->second.end(),
        parser::make_header_qvalue<string::const_iterator>(
            req->accept_encoding.get(),
            boost::bind(&parser::parse_content_encoding_ref<string::const_iterator>, _1, _2, _3)));
}

}
