#pragma once

#include <boost/enable_shared_from_this.hpp>
#include <boost/thread/mutex.hpp>
#include <yplatform/util/atomic_count.h>
#include <yplatform/log.h>
#include <ymod_webserver/websocket.h>
#include "contains_creation_time.h"
#include "session.h"
#include "parser/websocket.h"
#include "net_session.h"

namespace ymod_webserver {

using boost::optional;

class net_server;

namespace websocket {

// TODO rewrite to isolate message prepare functions

typedef yplatform::future::promise<void> promise_void_t;

class websocket_stream
    : public output_stream
    , public contains_creation_time
    , public yplatform::log::contains_logger
    , public boost::enable_shared_from_this<websocket_stream>
{
    typedef read_buffer_t::iterator read_iterator;
    typedef parser::websocket<read_iterator> websocket_parser_t;

public:
    typedef output_stream::message_hook_t message_hook_t;
    typedef output_stream::close_hook_t close_hook_t;
    websocket_stream(
        boost::weak_ptr<net_server> owner,
        context_ptr context,
        session_ptr session,
        const request_ptr& req,
        read_buffer_ptr readq,
        const yplatform::log::source& logger,
        size_t max_length,
        size_t max_fragmentation);
    ~websocket_stream();
    void init();

    void begin_receive(const yplatform::time_traits::duration& inactive_timeout) override
    {
        session_->current_io_service().dispatch(boost::bind(
            &websocket_stream::start_read_message_strand, shared_from_this(), inactive_timeout));
    }

    future_void_t send_text(const string& msg, uint8_t opcode) override
    {
        promise_void_t prom;
        session_->current_io_service().dispatch(boost::bind(
            &websocket_stream::send_text_strand, shared_from_this(), msg, opcode, prom));
        return prom;
    }
    future_void_t send_binary(const string& msg, uint8_t opcode) override
    {
        promise_void_t prom;
        session_->current_io_service().dispatch(boost::bind(
            &websocket_stream::send_binary_strand, shared_from_this(), msg, opcode, prom));
        return prom;
    }

    void close_connection(uint16_t code, const string& reason = "") override
    {
        session_->current_io_service().dispatch(boost::bind(
            &websocket_stream::handle_close_by_application, shared_from_this(), code, reason));
    }
    yplatform::time_traits::timer_ptr make_timer() const override;
    boost::asio::io_service& get_io_service() override;
    yplatform::net::streamer_wrapper text_stream(std::size_t sz, uint8_t opcode) override;
    yplatform::net::streamer_wrapper bin_stream(std::size_t sz, uint8_t opcode) override;

    void send_client_stream(yplatform::net::buffers::const_chunk_buffer const& s) override
    {
        session_->current_io_service().dispatch(
            boost::bind(&websocket_stream::send_client_stream_strand, shared_from_this(), s));
    }

    void add_message_callback(message_hook_t const& hook) override
    {
        session_->current_io_service().dispatch(
            boost::bind(&websocket_stream::add_message_callback_strand, shared_from_this(), hook));
    }

    void set_close_callback(close_hook_t const& hook) override
    {
        session_->current_io_service().dispatch(
            boost::bind(&websocket_stream::set_close_callback_strand, shared_from_this(), hook));
    }

    codes::code result_code() const
    {
        return result_code_;
    }

    request_ptr request() const override
    {
        return request_;
    }

    net_session_ptr get_session()
    {
        return session_;
    }

private:
    void send_client_stream_strand(yplatform::net::buffers::const_chunk_buffer const& s);
    void add_message_callback_strand(message_hook_t const& hook);
    void set_close_callback_strand(close_hook_t const& hook);

    void send_text_strand(const string& msg, uint8_t opcode, promise_void_t prom);
    void send_binary_strand(const string& msg, uint8_t opcode, promise_void_t prom);

    void close_connection_impl(uint16_t code, const string& reason);

    void handle_io_error(boost::system::error_code const& err, const char* operation);
    void handle_close_by_remote_peer();
    void handle_close_by_application(uint16_t code, const string& reason);

    void handle_runtime_error(const string& reason);

    void start_read_message_strand(const yplatform::time_traits::duration& inactive_timeout);
    void begin_read_message(std::size_t min = 1);

    void handle_read_message(boost::system::error_code const& err, std::size_t bytes);
    void process_read_buffer();

    optional<message> parse_message();

    mask_t generate_mask() const;
    void send_message_strand(
        const string& msg,
        uint8_t opcode,
        uint8_t bits,
        const mask_t& mask,
        bool apply_mask = false,
        promise_void_t prom = promise_void_t());

    void out_handshake();
    bool is_open() const override;
    bool is_secure() const override;
    string get_websocket_location();
    void validate_message(message const& msg);
    void perform_message(message const& msg);
    void on_receive(message const& sg);
    void deliver_message(const message& msg);
    void maybe_notify_closed(uint16_t code, const string& reason);

    boost::weak_ptr<websocket_stream> weak_from_this()
    {
        return shared_from_this();
    }

    boost::weak_ptr<net_server> owner_;
    context_ptr context_;
    session_ptr session_;
    request_ptr request_;
    read_buffer_ptr readq_;
    read_iterator i_saved_;
    websocket_parser_t parser_;
    optional<fragmented_message_state> fragmented_message_;
    codes::code result_code_ = codes::shutdown;

    std::vector<message_hook_t> message_hooks_;
    close_hook_t close_hook_;
    bool is_reading_;
    bool close_frame_sent_;
    size_t max_fragmentation_;
};

}
}
