#pragma once

#include <chrono>
#include <tuple>
#include <logdog/logger.h>
#include <logdog/format/tskv.h>
#include <logdog/backend/yplatform_log.h>
#include <macs_pg/subscription/subscription.h>
#include <macs_pg/subscribed_folders/subscribed_folder.h>
#include <macs/folder.h>
#include <macs_pg/macs_pg.h>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif

#include <ymod_httpclient/call.h>

#ifdef __clang__
#pragma clang diagnostic pop
#endif

#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptors.hpp>
#include <macs_pg/subscription/unsubscribe_task.h>

namespace york {
namespace log {

constexpr static auto york_formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-york-log"));

LOGDOG_DEFINE_ATTRIBUTE(std::string, session_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, request_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, endpoint)
LOGDOG_DEFINE_ATTRIBUTE(std::string, response)
LOGDOG_DEFINE_ATTRIBUTE(macs::Uid, owner_uid)
LOGDOG_DEFINE_ATTRIBUTE(macs::Uid, subscriber_uid)
LOGDOG_DEFINE_ATTRIBUTE(std::vector<macs::Fid>, owner_fids)
LOGDOG_DEFINE_ATTRIBUTE(macs::Fid, root_subscribed_fid)
LOGDOG_DEFINE_ATTRIBUTE(macs::Fid, fid)
LOGDOG_DEFINE_ATTRIBUTE(std::vector<macs::Fid>, fids)
LOGDOG_DEFINE_ATTRIBUTE(std::string, task_request_id)

using namespace logdog;

inline auto makeLog(const std::string& logId = "york") {
    auto ylog = std::make_shared<::yplatform::log::source>(
                YGLOBAL_LOG_SERVICE, logId);
    return ::logdog::make_log(york_formatter, ylog);
}

inline auto makeContextLog(const std::string& sessionId,
                           const std::string& requestId,
                           const std::string& logId = "york") {
    return ::logdog::bind(makeLog(logId),
            log::session_id=sessionId, log::request_id=requestId);
}


} // namespace log
} // namespace york

#define YORK_LOG_WITH_LEVEL(logger, level_name, ...) \
        ::york::log::level_name(logger, [&](auto __write) mutable {\
            __write(::york::log::message=__VA_ARGS__);\
        })

#define YORK_LOG_ERROR(logger, ...) YORK_LOG_WITH_LEVEL(logger, error, __VA_ARGS__)
#define YORK_LOG_WARNING(logger, ...) YORK_LOG_WITH_LEVEL(logger, warning, __VA_ARGS__)
#define YORK_LOG_NOTICE(logger, ...) YORK_LOG_WITH_LEVEL(logger, notice, __VA_ARGS__)
#define YORK_LOG_DEBUG(logger, ...) YORK_LOG_WITH_LEVEL(logger, debug, __VA_ARGS__)

#define STRING_LINE_I_(x) #x
#define STRING_LINE(x) STRING_LINE_I_(x)
#define YORK_WHERE ::std::make_tuple(\
        ::york::log::where_name=__FUNCTION__,\
        ::york::log::where_file=__FILE__,\
        ::york::log::where_line=STRING_LINE(__LINE__))

#define YORK_LOG_WITH_LEVEL_WHERE(logger, level_name, ...) \
        ::york::log::level_name(logger, [&, _w_ = YORK_WHERE](auto __write) mutable {\
            __write(::york::log::message=__VA_ARGS__, _w_);})

#define YORK_LOG_ERROR_WHERE(logger, ...) YORK_LOG_WITH_LEVEL_WHERE(logger, error, __VA_ARGS__)
#define YORK_LOG_WARNING_WHERE(logger, ...) YORK_LOG_WITH_LEVEL_WHERE(logger, warning, __VA_ARGS__)
#define YORK_LOG_NOTICE_WHERE(logger, ...) YORK_LOG_WITH_LEVEL_WHERE(logger, notice, __VA_ARGS__)
#define YORK_LOG_DEBUG_WHERE(logger, ...) YORK_LOG_WITH_LEVEL_WHERE(logger, debug, __VA_ARGS__)
