#pragma once

#include <internal/logger/logger.h>
#include <internal/config.h>
#include <internal/sharpei.h>

#include <macs_pg/service/service.h>
#include <macs_pg/macs_pg.h>

#include <yplatform/reactor.h>

namespace york {

template <typename Logger>
macs::pg::logging::v2::LogPtr getMacsPgLogger(Logger logger) {
    return ::pgg::logging::makeTypedLog(log::bind(logger, log::message="macs_pg"));
}

template <typename Logger>
inline auto createUidResolverFactory(const PgCfg& cfg, Logger log) {
    return macs::pg::createSharpeiUidResolverFactory(macs::pg::SharpeiParams{
            .httpClient=nullptr,
            .settings=sharpei::makeSharpeiParams(cfg.sharpei, log, "uid resolver"),
            .credentials={cfg.user, ""}
    });
}

template <typename Logger>
inline auto createShardResolverFactory(const PgCfg& cfg, Logger log) {
    return macs::pg::createShardResolverFactory(macs::pg::SharpeiParams{
            .httpClient=nullptr,
            .settings=sharpei::makeSharpeiParams(cfg.sharpei, log, "shard resolver"),
            .credentials={cfg.user, ""}
    });
}

class SuperPool {
    auto createPool(const PoolCfg& cfg, boost::asio::io_service& io);
    std::unordered_map<boost::asio::io_service*, macs::pg::ConnectionPoolPtr> pools;
public:
    void init(const PoolCfg& cfg);

    template <typename Logger>
    macs::ServicePtr getMacsService(const std::string& uid, Logger logger, ConfigPtr cfg) {
        auto factory = getFactory(logger, cfg);
        return factory->mailbox(uid);
    }

    template <typename Logger>
    macs::ShardPtr getMacsShard(const std::string& shardId, Logger logger, ConfigPtr cfg) {
        auto factory = getFactory(logger, cfg);
        return factory->shard(shardId);
    }

private:
    template <typename Logger>
    auto getFactory(Logger logger, ConfigPtr cfg) {
        using Ms = macs::pg::Milliseconds;
        auto factory = macs::pg::createSeviceFactory(
                pools.at(yplatform::global_net_reactor->get_pool()->io()),
                createUidResolverFactory(cfg->pg, logger),
                createShardResolverFactory(cfg->pg, logger));

        factory->queryConf(cfg->queryConf)
                .credentials({cfg->pg.user, ""})
                .transactionTimeout(Ms(cfg->pg.transaction_timeout_ms))
                .logger(getMacsPgLogger(logger));

        return factory;
    }
};

} // namespace york
