#pragma once

#include <internal/server/handlers/helpers.h>
#include <boost/fusion/include/count.hpp>

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), CreateListParams,
        (std::string, uid)
        (boost::optional<std::string>, shared_folder_name) )

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), CreateListResult,
        (std::string, shared_folder_fid) )

namespace york {
namespace server {
namespace handlers {

template<class ServiceT, class ContextT, class LoggerT, class YieldT>
void executeMacsCreateList(ServiceT& mailbox,
                           ContextT& ctx,
                           boost::optional<std::string> name,
                           FolderArchivation cfg,
                           LoggerT logger,
                           YieldT myield) {
    const auto folders = mailbox.folders().getAllFolders(myield);

    bool folderAlreadyExist = false;
    macs::Folder folder;
    if (!name) {
        folder = folders.at(macs::Folder::Symbol::inbox);
    } else if (!folders.exists(*name, macs::Folder::noParent)) {
        folder = mailbox.folders().createFolder(*name, myield);
    } else {
        folder = folders.at(*name, macs::Folder::noParent);
        folderAlreadyExist = true;
    }

    const auto sharedFoldersFids = mailbox.sharedFolders().getAllSharedFolders(myield);
    if (boost::count(sharedFoldersFids, folder.fid())) {
        ctx.response().ok(CreateListResult{folder.fid()});
        return;
    }

    if (folderAlreadyExist && folder.messagesCount()) {
        ctx.response().badRequest(
                    Error{"folder already exisis and is not empty"},
                    logger,
                    log::fid=folder.fid());
        return;
    }

    mailbox.sharedFolders().createSharedFolderWithArchivation(folder.fid(),
            macs::Folder::ArchivationType::fromString(cfg.type),
            cfg.keep_days, cfg.max_folder_size, myield);
    ctx.response().ok(CreateListResult{folder.fid()});
}

template<typename ConfigPtr, typename MacsGetter>
auto makeCreateListHandler(ConfigPtr cfg, MacsGetter getMacsService) {
    return makeHandler([cfg, getMacsService](auto& ctx, auto log, auto yield) {
        CreateListParams params;
        if (!getArgs(ctx, params, log)) {
            return;
        }
        auto mailbox = getMacsService(params.uid, log, cfg);
        executeMacsCreateList(*mailbox, ctx, params.shared_folder_name,
                              cfg->archivation, log, wrap(yield));
    });
}

}
}
}
