#pragma once

#include <internal/config.h>
#include <internal/logger/logger.h>

#include <yamail/data/reflection.h>
#include <yamail/data/deserialization/yajl.h>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif

#include <yplatform/reactor.h>

#ifdef __clang__
#pragma clang diagnostic pop
#endif

#include <macs/error_code.h>
#include <macs/io.h>
#include <boost/asio/spawn.hpp>
#include <boost/system/system_error.hpp>
#include <boost/fusion/adapted/struct/define_struct.hpp>

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), Error,
                           (std::string, error) )

namespace york {
namespace server {
namespace handlers {

using error_code = macs::sys::error_code;

inline auto wrap(boost::asio::yield_context yield) {
    return ::macs::io::make_yield_context(yield);
}

template<typename Action>
auto makeHandler(Action act) {
    return [act = std::move(act)](auto ctx, auto log) mutable {
        auto io = yplatform::global_net_reactor->io();
        boost::asio::spawn(*io, [ctx, log, act] (auto yield) {
            try {
                act(*ctx, log, yield);
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const boost::system::system_error& e) {
                const auto msg = e.code().category().name() + std::string(" ") + std::string(e.what());
                ctx->response().internalError(Error{msg}, log);
            } catch (const std::exception& e) {
                const auto msg = std::string(e.what());
                ctx->response().internalError(Error{msg}, log);
            }
        });
    };
}

template<typename ContextT>
struct Visitor : public yreflection::Visitor {
    Visitor(ContextT& ctx): ctx(ctx) {}
    ContextT& ctx;

    template <typename Struct, typename Tag>
    Visitor onStructStart(Struct&&, Tag) { return *this; }

    template <typename T, typename Tag>
    Visitor onSequenceStart(std::vector<T>& v, Tag tag) {
        const std::string& name = yreflection::name(tag);
        const auto opt = ctx.getOptionalArg(name);
        if (!opt) {
            throw std::invalid_argument(name + " required");
        }
        yamail::data::deserialization::fromJson(*opt, v);
        return *this;
    }

    template <typename Value, typename ... Args>
    void onValue(Value& v, yreflection::NamedItemTag<Args...> tag) {
        const std::string& name = yreflection::name(tag);
        auto opt = ctx.getOptionalArg(name);
        if( !opt ) {
            throw std::invalid_argument(name + " required");
        }

        if (!boost::conversion::try_lexical_convert(std::move(*opt), v)) {
            throw std::invalid_argument(std::string("can not convert ") + name);
        }
    }

    template <typename Value>
    void onValue(Value&, yreflection::SequenceItemTag) {}

    template <typename ... Args>
    void onValue(bool& v, yreflection::NamedItemTag<Args...> tag) {
        const std::string name = yreflection::name(tag);
        auto opt = ctx.getOptionalArg(name);
        if( !opt ) {
            throw std::invalid_argument(name + " required");
        }
        if (*opt == "yes") {
            v = true;
        } else if (*opt == "no") {
            v = false;
        } else {
            throw std::invalid_argument(name + " should be 'yes' or 'no'");
        }
    }

    template <typename Value, typename ... Args>
    bool onOptional(boost::optional<Value>& v, yreflection::NamedItemTag<Args...> tag) {
        const std::string& name = yreflection::name(tag);
        auto opt = ctx.getOptionalArg(name);
        if ( !opt ) {
            v = boost::none;
            return false;
        }

        Value val;
        if (!boost::conversion::try_lexical_convert(std::move(*opt), val)) {
            throw std::invalid_argument(std::string("can not convert ") + name);
        }

        v = boost::make_optional(val);
        return false;
    }

    template <typename ... Args>
    bool onOptional(boost::optional<bool>& v, yreflection::NamedItemTag<Args...> tag) {
        const std::string name = yreflection::name(tag);
        const auto arg = ctx.getOptionalArg(name);
        if ( !arg ) {
            v = boost::none;
        } else if (*arg == "yes") {
            v = boost::make_optional(true);
        } else if (*arg == "no") {
            v = boost::make_optional(false);
        } else {
            throw std::invalid_argument(name + " should be 'yes' or 'no'");
        }
        return false;
    }
};

template <typename ContextT, typename ParamsT, typename LoggerT = log::none_t>
bool getArgs(ContextT& ctx, ParamsT& out, LoggerT logger = log::none) {
    Visitor<ContextT> visitor(ctx);
    try{
        yreflection::applyVisitor(out, visitor, yreflection::namedItemTag("root"));
        return true;
    } catch( const std::invalid_argument& e ) {
        ctx.response().badRequest(Error{e.what()}, logger);
    }
    return false;
}

} //namespace handers
} //namespace server
} //namespace york
