#pragma once

#include <internal/server/handlers/helpers.h>
#include <boost/range/algorithm.hpp>

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), SetArchivationRuleParams,
                           (std::string, uid)
                           (boost::optional<std::string>, shared_folder_fid)
                           (std::string, type)
                           (boost::optional<uint32_t>, keep_days)
                           (boost::optional<uint32_t>, max_size) )

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), SetArchivationRuleResult,
                           (bool, ok) )

namespace york {
namespace server {
namespace handlers {

template<class ServiceT, class ContextT, class LoggerT, class YieldT>
void executeMacsSetArchivationRule(ServiceT& mailbox,
                                   ContextT& ctx,
                                   SetArchivationRuleParams params,
                                   FolderArchivation cfg,
                                   LoggerT logger,
                                   YieldT myield) {
    const auto sharedFolderFid = params.shared_folder_fid
                      ? *params.shared_folder_fid
                      : mailbox.folders().getAllFolders(myield).fid(macs::Folder::Symbol::inbox);

    const auto sharedFolders = mailbox.sharedFolders().getAllSharedFolders(myield);
    if (boost::count(sharedFolders, sharedFolderFid) == 0) {
        ctx.response().badRequest(
                    Error{"folder is not shared or does not exist, fid: " + sharedFolderFid},
                    logger,
                    log::owner_uid=params.uid,
                    log::fid=sharedFolderFid);
        return;
    }

    const auto type = macs::Folder::ArchivationType::fromString(params.type, std::nothrow);
    if (type == macs::Folder::ArchivationType::unknown) {
        ctx.response().badRequest(
                    Error{"unknown archivation type: " + params.type},
                    logger);
        return;
    }

    const auto min = [](boost::optional<uint32_t> param, uint32_t cfg) {
        return param ? std::min(*param, cfg) : cfg;
    };

    mailbox.folders().setArchivationRule(sharedFolderFid, type,
            min(params.keep_days, cfg.keep_days),
            min(params.max_size, cfg.max_folder_size),
            myield);

    ctx.response().ok(SetArchivationRuleResult{true});
}

template<typename ConfigPtr, typename MacsGetter>
auto makeSetArchivationRuleHandler(ConfigPtr cfg, MacsGetter getMacsService) {
    return makeHandler([cfg, getMacsService](auto& ctx, const auto& log, auto yield) {
        SetArchivationRuleParams params;
        if (!getArgs(ctx, params)) {
            return;
        }
        auto mailbox = getMacsService(params.uid, log, cfg);
        executeMacsSetArchivationRule(*mailbox, ctx, std::move(params), cfg->archivation, log, wrap(yield));
    });
}

}
}
}
