#pragma once

#include <internal/server/handlers/helpers.h>
#include <internal/logger/logger.h>
#include <boost/range/adaptors.hpp>
#include <boost/range/join.hpp>
#include <set>
#include <macs_pg/subscription/subscription_state.h>

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), SubscribeParams,
                           (std::string, subscriber_uid)
                           (std::vector<std::string>, destination_folder_path)
                           (std::string, owner_uid)
                           (boost::optional<std::string>, shared_folder_fid)
                           (boost::optional<bool>, recursive)
                           (boost::optional<bool>, imap_unsubscribed) )

BOOST_FUSION_DEFINE_STRUCT((york)(server)(handlers), SubscribeResult,
                           (bool, ok) )

namespace york {
namespace server {
namespace handlers {

using FidPath = std::tuple<macs::Fid, macs::Folder::Path>;
using FidPathVec = std::vector<FidPath>;

template<class ServiceT, class YieldT>
auto getSharedSubfoldersPath(const ServiceT& ownerMailbox, const macs::Fid& sharedFolderFid,
                             const macs::Folder::Path& destPath, bool recursive, YieldT myield) {

    FidPathVec sharedSubfolders;
    const auto sharedFids = ownerMailbox.sharedFolders().getAllSharedFolders(myield);
    if (boost::count(sharedFids, sharedFolderFid) == 0) {
        return sharedSubfolders;
    }

    sharedSubfolders.emplace_back(sharedFolderFid, destPath);

    if (recursive) {
        const auto ownerFolders = ownerMailbox.folders().getAllFolders(myield);

        const auto rootFolder = ownerFolders.at(sharedFolderFid);
        std::int32_t rootPathLength = static_cast<std::int32_t>(
            ownerFolders.getPath(rootFolder).size());

        auto sharedOnly = boost::adaptors::filtered([&](const auto& f) {
            return boost::count(sharedFids, f.fid());
        });

        auto convertPath = boost::adaptors::transformed([&](const auto& f) {
            macs::Folder::Path ownerPath = ownerFolders.getPath(f);
            macs::Folder::Path subFolder(ownerPath.begin() + rootPathLength, ownerPath.end());
            return std::make_tuple(f.fid(), macs::Folder::Path(boost::range::join(destPath, subFolder)));
        });

        boost::copy(ownerFolders.getChildrenRecursive(rootFolder) | sharedOnly | convertPath,
                    std::back_inserter(sharedSubfolders));
    }

    return sharedSubfolders;
}

inline bool needCreate(const FidPath& sharedSubfolder,
            const std::vector<macs::SubscribedFolder> subscribedFolders) {
    return !(boost::count_if(subscribedFolders, [&](const auto& sf){
        return sf.ownerFid() == std::get<macs::Fid>(sharedSubfolder);
    }));
}

inline std::vector<macs::Fid> getExistingFids(const FidPathVec& sharedSubfolders,
            const macs::FolderSet& userFolders,
            const std::vector<macs::SubscribedFolder>& subscribedFolders) {
    using namespace boost::adaptors;

    auto subscriptionFree = [&](auto& f){
        return needCreate(f, subscribedFolders);
    };
    auto found = [&](auto& f){
        auto& path = std::get<macs::Folder::Path>(f);
        return userFolders.find(path);
    };
    auto hasMessages = [&](auto& f){
        return f != userFolders.end() && f->second.messagesCount() > 0;
    };

    const auto existingFids = sharedSubfolders
            | filtered(subscriptionFree)
            | transformed(found)
            | filtered(hasMessages)
            | indirected | map_keys;
    return {existingFids.begin(), existingFids.end()};
}

template<class ServiceT, class YieldT>
inline std::vector<macs::Subscription> getTerminating(
            const ServiceT& ownerMailbox,
            const FidPathVec& sharedSubfolders,
            const std::string subscriberUid,
            YieldT myield) {
    using namespace boost::adaptors;

    auto sharedFids = sharedSubfolders
            | transformed([](auto& f){ return std::get<macs::Fid>(f); });

    auto existingSubs = ownerMailbox.subscriptions().getByFids(subscriberUid,
            {std::begin(sharedFids), std::end(sharedFids)}, myield);

    auto terminatingSubs = existingSubs | filtered([](auto& s) {
        return s.state() == macs::pg::SubscriptionState::discontinued ||
               s.state() == macs::pg::SubscriptionState::clear ||
               s.state() == macs::pg::SubscriptionState::clearFail ||
               s.state() == macs::pg::SubscriptionState::terminated;
    });
    return {std::begin(terminatingSubs), std::end(terminatingSubs)};
}

template<class ServiceT, class ContextT, class LoggerT, class YieldT>
void executeMacsSubscribe(const ServiceT& ownerMailbox, const ServiceT& subscriberMailbox,
                          ContextT& ctx, const SubscribeParams& params, LoggerT logger, YieldT myield) {
    if(params.destination_folder_path.empty()) {
        ctx.response().badRequest(
                    Error{"destination_folder_path is empty"},
                    logger);
        return;
    }

    const auto sharedFolderFid = params.shared_folder_fid
                      ? *params.shared_folder_fid
                      : ownerMailbox.folders().getAllFolders(myield).fid(macs::Folder::Symbol::inbox);

    bool recursive = params.recursive && *params.recursive;

    auto sharedSubfolders = getSharedSubfoldersPath(ownerMailbox, sharedFolderFid,
                                                    macs::Folder::Path(params.destination_folder_path),
                                                    recursive, myield);
    if (sharedSubfolders.empty()) {
        ctx.response().badRequest(
                    Error{"folder is not shared or does not exist, fid: "
                          + sharedFolderFid},
                    logger,
                    log::owner_uid=params.owner_uid,
                    log::fid=sharedFolderFid);
        return;
    }

    const auto terminatingSubs = getTerminating(ownerMailbox, sharedSubfolders,
                                                params.subscriber_uid, myield);
    if (!terminatingSubs.empty()) {
        const auto subFids = terminatingSubs | boost::adaptors::transformed([](auto& s) {
            return s.fid();
        });
        ctx.response().badRequest(
                    Error{"subscriptions still terminating, fids: " +
                          boost::algorithm::join(subFids, ",")},
                    logger,
                    log::owner_uid=params.owner_uid,
                    log::subscriber_uid=params.subscriber_uid,
                    log::owner_fids={subFids.begin(), subFids.end()});
        return;
    }

    const auto userFolders = subscriberMailbox.folders().getAllFolders(myield);
    const auto subscribedFolders = subscriberMailbox.subscribedFolders()
            .getFoldersByOwner(params.owner_uid, myield);

    const auto existingFids = getExistingFids(sharedSubfolders, userFolders, subscribedFolders);
    if (!existingFids.empty()) {
        ctx.response().badRequest(
                    Error{"folders already exist and not empty, fids:" +
                          boost::algorithm::join(existingFids, ",")},
                    logger,
                    log::subscriber_uid=params.subscriber_uid,
                    log::fids=existingFids);
        return;
    }

    boost::for_each(sharedSubfolders, [&](auto& f) mutable {
        if (needCreate(f, subscribedFolders)) {
            auto& path = std::get<macs::Folder::Path>(f);
            const auto folder = subscriberMailbox.folders().createFolderByPath(path, myield);
            subscriberMailbox.subscribedFolders()
                    .addFolder(folder.fid(), params.owner_uid, std::get<macs::Fid>(f), myield);
            if (params.imap_unsubscribed && *params.imap_unsubscribed) {
                try {
                    subscriberMailbox.imapRepo().imapUnsubscribeFolder({path.begin(), path.end()}, myield);
                } catch (const std::exception& e) {
                    YORK_LOG_ERROR(logger, "imap unsubscribe error",
                                   log::exception=e);
                }
            }
        }
        ownerMailbox.sharedFolders()
                .addSubscriber(std::get<macs::Fid>(f), params.subscriber_uid, myield);
    });

    try {
        subscriberMailbox.imapRepo().imapUnsubscribeFolder({"Yandex"}, myield);
    } catch (const std::exception& e) {
        YORK_LOG_ERROR(logger, "imap unsubscribe error",
                       log::exception=e);
    }

    ctx.response().ok(SubscribeResult{true});
}

template<typename MacsGetter>
auto makeSubscribeHandler(ConfigPtr cfg, MacsGetter getMacsService) {
    return makeHandler([cfg, getMacsService](auto& ctx, auto log, auto yield) {
        SubscribeParams params;
        if (!getArgs(ctx, params, log)) {
            return;
        }

        auto subscriberMailbox = getMacsService(params.subscriber_uid, log, cfg);
        auto ownerMailbox = getMacsService(params.owner_uid, log, cfg);

        executeMacsSubscribe(*ownerMailbox, *subscriberMailbox, ctx, params, log, wrap(yield));
    });
}

}
}
}
