#pragma once

#include <internal/server/operations/unsubscribe_operation.h>
#include <internal/logger/logger.h>
#include <macs_pg/subscription/unsubscribe_task_factory.h>

namespace york {
namespace server {
namespace handlers {
namespace operations {

using boost::adaptors::transformed;
using boost::adaptors::filtered;

template <typename Service,
          typename Context,
          typename Logger>
class UnsubscribeOperationImpl {
public:
    UnsubscribeOperationImpl(const Service& ownerMailbox_,
                         const Service& subscriberMailbox_,
                         Context& ctx_,
                         const UnsubscribeParams& params_,
                         Logger log_)
        : ownerMailbox(ownerMailbox_)
        , subscriberMailbox(subscriberMailbox_)
        , ctx(ctx_)
        , params(params_)
        , log(std::move(log_)) {}

    template <typename Myield>
    SubsFolderVec getSubfolders(const macs::Fid& sharedFolderFid,
                                Myield myield) const {
        SubsFolderVec subscribedSubfolders;

        const auto userFolders = subscriberMailbox.folders().getAllFolders(myield);
        const auto subscribedFolders = subscriberMailbox.subscribedFolders()
                .getFoldersByOwner(params.owner_uid, myield);
        const auto rootSubscribed = getBySharedFid(subscribedFolders, sharedFolderFid);
        if (rootSubscribed != subscribedFolders.end()) {
            subscribedSubfolders.push_back(*rootSubscribed);
            boost::copy(userFolders.getChildrenRecursive(userFolders.at(rootSubscribed->fid()))
                            | transformed([&](const auto& f) { return this->getByFid(subscribedFolders, f.fid()); })
                            | filtered([&](const auto& it) { return it != subscribedFolders.end(); })
                            | boost::adaptors::indirected,
                        std::back_inserter(subscribedSubfolders));
        }
        return subscribedSubfolders;
    }

    bool operable(const macs::Subscription& s) const {
        return s.state() != macs::pg::SubscriptionState::terminated
                && s.state() != macs::pg::SubscriptionState::discontinued
                && s.state() != macs::pg::SubscriptionState::clear
                && s.state() != macs::pg::SubscriptionState::clearFail;
    }

    template <typename Myield>
    void terminate(const SubsFolderVec& folders,
                   Myield myield) const {
        const auto subscriptions = getByFolders(folders, myield);
        boost::for_each(subscriptions | filtered([&](const auto& s){ return this->operable(s); }),
                        [&](const auto& s){
            ownerMailbox.subscriptions().transitState(s.subscriptionId(),
                    macs::pg::SubscriptionAction::unsubscription, myield);
        });
    }

    template <typename Myield>
    void createUnsubscribeTask(const SubsFolderVec& folders,
                               Myield myield) const {
        macs::UnsubscribeTask task = macs::UnsubscribeTaskFactory()
                .taskRequestId(ctx.requestId())
                .ownerUid(params.owner_uid)
                .ownerFids(getSharedFids(folders))
                .subscriberUid(params.subscriber_uid)
                .rootSubscriberFid(folders.front().fid())
                .release();

        task = ownerMailbox.subscriptions().addUnsubscribeTask(std::move(task), myield);

        YORK_LOG_NOTICE(log, "Add unsubscribe task",
                        log::task_request_id=task.taskRequestId(),
                        log::owner_uid=task.ownerUid(),
                        log::subscriber_uid=task.subscriberUid(),
                        log::owner_fids=task.ownerFids(),
                        log::root_subscribed_fid=task.rootSubscriberFid());
    }

    template <typename Myield>
    void removeSubscription(const macs::Fid& sharedFolderFid,
                            Myield myield) const {
        SubscriptionVec subscriptions = ownerMailbox.subscriptions().getByFids(
                params.subscriber_uid, {sharedFolderFid}, myield);
        if (!subscriptions.empty()) {
            const auto subscriptionIds = subscriptions | transformed(
                    [](const auto& s){ return s.subscriptionId(); });
            ownerMailbox.subscriptions().removeChunk(
                    {subscriptionIds.begin(), subscriptionIds.end()},
                    myield);
        }
    }

    const Service& ownerMailbox;
    const Service& subscriberMailbox;
    Context& ctx;
    UnsubscribeParams params;
    Logger log;

private:
    auto getByFid(const SubsFolderVec& folders,
                  const macs::Fid& fid) const {
        return boost::find_if(folders, [&](const auto& sf){
            return sf.fid() == fid;
        });
    }

    auto getBySharedFid(const SubsFolderVec& folders,
                     const macs::Fid& sharedFid) const {
        return boost::find_if(folders, [&](const auto& sf){
            return sf.ownerFid() == sharedFid;
        });
    }

    FidVec getSharedFids(const SubsFolderVec& folders) const {
        const auto fids = folders | transformed([](const auto& sf){ return sf.ownerFid(); });
        return {fids.begin(), fids.end()};
    }

    template <typename Myield>
    SubscriptionVec getByFolders(const SubsFolderVec& folders,
                                 Myield myield) const {
        return ownerMailbox.subscriptions().getByFids(
                    params.subscriber_uid, getSharedFids(folders), myield);
    }

};

template <typename Service, typename Context, typename Logger>
inline auto makeUnsubscribeOperationImpl(const Service& ownerMailbox,
                                     const Service& subscriberMailbox,
                                     Context& ctx,
                                     const UnsubscribeParams& params,
                                     Logger log) {
    using Impl = operations::UnsubscribeOperationImpl<Service, Context, Logger>;

    return std::make_shared<Impl>(ownerMailbox, subscriberMailbox,
                                  ctx, params, std::move(log));
}
} // namespace operations

template <typename Service, typename Context, typename Logger>
inline auto makeUnsubscribeOperation(const Service& ownerMailbox,
                                     const Service& subscriberMailbox,
                                     Context& ctx,
                                     const UnsubscribeParams& params,
                                     Logger log) {
    auto impl = operations::makeUnsubscribeOperationImpl(
                ownerMailbox, subscriberMailbox, ctx, params, std::move(log));
    return operations::makeUnsubscribeOperationWithImpl(std::move(impl));
}
} // namespace handlers
} // namespace server
} // namespace york
