#pragma once

#include <ymod_webserver/response.h>
#include <yamail/data/serialization/json_writer.h>
#include <internal/logger/logger.h>

namespace york {
namespace server {

struct Response {
    Response(ymod_webserver::response_ptr component)
            : response_(component) {}

    template <class Result>
    void ok(const Result& result) const {
        response_->set_code(ymod_webserver::codes::ok);
        write(result);
    }

    template <class Result, typename Logger, typename ... LogArgs>
    void badRequest(const Result& result,
                    Logger logger,
                    LogArgs&& ... logArgs) const {
        response_->set_code(ymod_webserver::codes::bad_request);
        const auto resp = write(result);
        YORK_LOG_WARNING(logger, "Bad request",
                         log::response=resp,
                         std::forward<LogArgs>(logArgs)...);
    }

    template <class Result>
    void badRequest(const Result& result) const {
        badRequest(result, log::none);
    }

    template <class Result, typename Logger, typename ... LogArgs>
    void internalError(const Result& result,
                       Logger logger,
                       LogArgs&& ... logArgs) const {
        response_->set_code(ymod_webserver::codes::internal_server_error);
        const auto resp = write(result);
        YORK_LOG_WARNING(logger, "Internal error",
                         log::response=resp,
                         std::forward<LogArgs>(logArgs)...);
    }

    template <class Result>
    void internalError(const Result& result) const {
        internalError(result, log::none);
    }

    template <class Result, typename Logger, typename ... LogArgs>
    void wrongMethod(const Result& result,
                     Logger logger,
                     LogArgs&& ... logArgs) const {
        response_->set_code(ymod_webserver::codes::method_not_allowed);
        const auto resp = write(result);
        YORK_LOG_WARNING(logger, "Wrong method",
                         log::response=resp,
                         std::forward<LogArgs>(logArgs)...);
    }

    template <class Result>
    void wrongMethod(const Result& result) const {
        wrongMethod(result, log::none);
    }

    template <class Result, typename Logger, typename ... LogArgs>
    void unauthorized(const Result& result,
                      Logger logger,
                      LogArgs&& ... logArgs) const {
        response_->set_code(ymod_webserver::codes::unauthorized);
        const auto resp = write(result);
        YORK_LOG_WARNING(logger, "Unauthorized",
                         log::response=resp,
                         std::forward<LogArgs>(logArgs)...);
    }

    template <class Result>
    void unauthorized(const Result& result) const {
        unauthorized(result, log::none);
    }

private:
    ymod_webserver::response_ptr response_;

    template <class Result>
    auto write(const Result& result) const {
        const auto resp = yamail::data::serialization::toJson(result).str();
        response_->set_content_type("application", "json");
        response_->result_body(resp);
        return resp;
    }
};

}
}
