#pragma once

#include <internal/timer.h>
#include <internal/config.h>
#include <internal/logger/logger.h>
#include <internal/server/handlers/helpers.h>

namespace york {
namespace worker {

using UnsubscribeTask = ::macs::UnsubscribeTask;

namespace operations {

template <typename Service, typename Logger>
class Unsubscribe {
public:
    Unsubscribe(const Service& ownerMailbox,
                const Service& subscriberMailbox,
                UnsubscribeTask task,
                Logger logger)
            : ownerMailbox_(ownerMailbox)
            , subscriberMailbox_(subscriberMailbox)
            , task_(std::move(task))
            , logger_(std::move(logger)) {
    }

    template <typename Myield>
    void run(Myield myield) const {
        removeSubscribedFolders(myield);
        removeUserFolders(myield);
        removeSubscriptionsAndTask(myield);
    }

private:
    const Service& ownerMailbox_;
    const Service& subscriberMailbox_;
    UnsubscribeTask task_;
    Logger logger_;

    template <typename Myield>
    void removeSubscribedFolders(Myield myield) const {
        YORK_LOG_NOTICE(logger_, "Will remove subscribed folders",
                        log::owner_uid=task_.ownerUid(),
                        log::subscriber_uid=task_.subscriberUid(),
                        log::owner_fids=task_.ownerFids());

        subscriberMailbox_.subscribedFolders().removeFolders(
                    task_.ownerUid(), task_.ownerFids(), myield);

        YORK_LOG_NOTICE(logger_, "Removed subscribed folders");
    }

    template <typename Myield>
    void removeUserFolders(Myield myield) const {
        YORK_LOG_NOTICE(logger_, "Will remove user folders",
                        log::subscriber_uid=task_.subscriberUid(),
                        log::root_subscribed_fid=task_.rootSubscriberFid());

        const auto& repo = subscriberMailbox_.folders();
        repo.resetFoldersCache();
        const auto userFolders = repo.getAllFolders(myield);
        const auto found = userFolders.find(task_.rootSubscriberFid());

        if (found != userFolders.end()) {
            const macs::Folder& folder = found->second;
            std::vector<macs::Fid> haveMessages;
            if (folder.messagesCount() > 0) {
                haveMessages.push_back(folder.fid());
            }
            boost::copy(userFolders.getChildrenRecursive(folder)
                            | boost::adaptors::filtered([](const auto& f){ return f.messagesCount() > 0; })
                            | boost::adaptors::transformed([](const auto& f){ return f.fid(); }),
                        std::back_inserter(haveMessages));
            if (!haveMessages.empty()) {
                YORK_LOG_WARNING(logger_, "Cannot remove user folders with messages",
                                 log::subscriber_uid=task_.subscriberUid(),
                                 log::fids=haveMessages);
                return;
            }

            repo.eraseCascade(folder, myield);
        }

        YORK_LOG_NOTICE(logger_, "Removed user folders");
    }

    template <typename Myield>
    void removeSubscriptionsAndTask(Myield myield) const {
        YORK_LOG_NOTICE(logger_, "Will remove subscriptions with task",
                        log::owner_uid=task_.ownerUid(),
                        log::subscriber_uid=task_.subscriberUid(),
                        log::owner_fids=task_.ownerFids(),
                        log::root_subscribed_fid=task_.rootSubscriberFid());

        ownerMailbox_.subscriptions().removeSubscriptionsAndTask(task_, myield);

        YORK_LOG_NOTICE(logger_, "Removed subscriptions");
    }
};

template <typename Service, typename Logger>
inline auto makeUnsubscribe(const Service& ownerMailbox,
                            const Service& subscriberMailbox,
                            UnsubscribeTask task,
                            Logger logger) {
    return Unsubscribe<Service, Logger>(ownerMailbox,
                                        subscriberMailbox,
                                        std::move(task),
                                        std::move(logger));
}

} // namespace operations

template <typename ConfigPtr, typename MacsGetter>
inline auto makeUnsubscribeSpawner(boost::asio::io_service& io, ConfigPtr cfg, const MacsGetter& getMacs) {
    return [&io, cfg, getMacs](const UnsubscribeTask& task) {
        boost::asio::spawn(io, [cfg, getMacs, task](auto yield) mutable {
            auto logger = log::makeContextLog("", task.taskRequestId(), "worker");
            YORK_LOG_NOTICE(logger, "will process task",
                            log::owner_uid=task.ownerUid(),
                            log::subscriber_uid=task.subscriberUid(),
                            log::owner_fids=task.ownerFids(),
                            log::root_subscribed_fid=task.rootSubscriberFid());
            try {
                auto ownerMailbox = getMacs(task.ownerUid(), logger, cfg);
                auto subscriberMailbox = getMacs(task.subscriberUid(), logger, cfg);

                auto singleUnsubscribe = operations::makeUnsubscribe(*ownerMailbox, *subscriberMailbox, task, logger);
                singleUnsubscribe.run(server::handlers::wrap(yield));
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const boost::system::system_error& e) {
                YORK_LOG_ERROR(logger, "Exception in singleUnsubscribe", log::exception=e);
            } catch (const std::exception& e) {
                YORK_LOG_ERROR(logger, "Exception in singleUnsubscribe", log::exception=e);
            }
        });
    };
}

} // namespace worker
} // namespace york

