#pragma once

#include <internal/timer.h>
#include <internal/config.h>
#include <internal/logger/logger.h>
#include <internal/server/handlers/helpers.h>
#include <internal/unsubscribe_worker/single_unsubscribe.h>
#include <internal/sharpei.h>

#include <macs_pg/subscription/factory.h>
#include <macs_pg/subscription/subscription_state.h>

#include <boost/range/algorithm_ext/push_back.hpp>
#include <boost/range/algorithm/random_shuffle.hpp>

namespace york {
namespace worker {

using server::handlers::wrap;

template <typename ConfigPtr,
          typename SharpeiClient,
          typename ShardGetter,
          typename UnsubscribeSpawner,
          typename Logger,
          typename Timer,
          typename Clock = std::chrono::steady_clock>
class UnsubscribeWorker {
    using UnsubscribeTaskVec = std::vector<UnsubscribeTask>;
    using Duration = typename Clock::duration;
    using Timepoint = typename Clock::time_point;
    using ShardId = ::sharpei::client::Shard::Id;

    static auto now() { return Clock::now(); }
public:
    UnsubscribeWorker(ConfigPtr cfg,
                      SharpeiClient sharpeiClient,
                      ShardGetter getShard,
                      UnsubscribeSpawner spawnUnsubscribe,
                      Logger logger,
                      Timer timer)
            : cfg_(cfg)
            , sharpeiClient_(std::move(sharpeiClient))
            , getShard_(std::move(getShard))
            , spawnUnsubscribe_(std::move(spawnUnsubscribe))
            , logger_(std::move(logger))
            , timer_(std::move(timer))
            , lastCalled_(now()) {}

    template <typename Yield>
    void run(Yield yield) {
        UnsubscribeTaskVec tasks = getTerminated(yield);
        if (!tasks.empty()) {
            YORK_LOG_NOTICE(logger_, std::string("Got tasks: ") + std::to_string(tasks.size()));
            for (const auto& task: tasks) {
                runSingleUnsubscribe(task);
            }
        }
    }

private:
    ConfigPtr cfg_;
    SharpeiClient sharpeiClient_;
    ShardGetter getShard_;
    UnsubscribeSpawner spawnUnsubscribe_;
    Logger logger_;
    Timer timer_;
    Timepoint lastCalled_;

    auto getShard(ShardId shardId) const {
        return getShard_(shardId, logger_, cfg_);
    }

    template <typename Myield>
    std::vector<ShardId> getShards(Myield myield) const {
        const auto stats = sharpeiClient_->stat(myield);
        const auto shards = stats | boost::adaptors::map_keys;

        std::vector<ShardId> result {std::begin(shards), std::end(shards)};
        boost::range::random_shuffle(result);
        return result;
    }

    Duration minDelay() const {
        return std::chrono::seconds(cfg_->worker.delay_sec);
    }

    std::size_t taskLimit() const {
        return static_cast<std::size_t>(cfg_->worker.max_task_count);
    }
    std::chrono::seconds taskTtl() const {
        return std::chrono::seconds(cfg_->worker.delay_sec * 2);
    }

    template <typename Yield>
    void holdDelay(Yield yield) {
        const auto delay = now() - lastCalled_;
        if (delay < minDelay()) {
            timer_.wait(minDelay() - delay, yield);
        }
        lastCalled_ = now();
    }

    template <typename Yield>
    UnsubscribeTaskVec getTerminated(Yield yield) {
        holdDelay(yield);

        UnsubscribeTaskVec tasks;
        auto shards = getShards(wrap(yield));
        for (const auto& shardId: shards) {
            auto shard = getShard(shardId);
            auto shardTasks = shard->subscriptions().getUnsubscribeTasks(
                        taskLimit() - tasks.size(), taskTtl(), wrap(yield));
            boost::range::push_back(tasks, shardTasks);
            if (tasks.size() >= taskLimit()) {
                break;
            }
        }
        return tasks;
    }

    void runSingleUnsubscribe(const UnsubscribeTask& task) const {
        spawnUnsubscribe_(task);
    }
};

template <typename ConfigPtr,
          typename SharpeiClient,
          typename ShardGetter,
          typename UnsubscribeSpawner,
          typename Logger,
          typename Timer>
inline auto makeUnsubscribeWorker(ConfigPtr cfg,
                                  SharpeiClient sharpeiClient,
                                  ShardGetter getShard,
                                  UnsubscribeSpawner spawner,
                                  Logger logger,
                                  Timer timer) {
    return UnsubscribeWorker<ConfigPtr, SharpeiClient, ShardGetter,
                             UnsubscribeSpawner, Logger, Timer>(
                cfg, std::move(sharpeiClient), std::move(getShard),
                std::move(spawner), std::move(logger), std::move(timer));
}

} // namespace worker

template <typename ConfigPtr,
          typename SharpeiGetter,
          typename MacsGetter,
          typename ShardGetter,
          typename RunStatus>
void runUnsubscribeWorker(boost::asio::io_service& io,
                          ConfigPtr cfg,
                          const SharpeiGetter& getSharpei,
                          const MacsGetter& getMacs,
                          const ShardGetter& getShard,
                          const RunStatus& running) {
    boost::asio::spawn(io, [&io, cfg, getSharpei, getMacs, getShard, &running](auto yield) mutable {
        auto logger = log::makeLog("worker");
        YORK_LOG_NOTICE(logger, "worker initialized");
        auto worker = worker::makeUnsubscribeWorker(
                cfg,
                getSharpei(cfg->pg.sharpei, logger),
                getShard,
                worker::makeUnsubscribeSpawner(io, cfg, getMacs),
                logger, timer::Timer(io)
        );

        while (running) {
            try {
                worker.run(yield);
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const boost::system::system_error& e) {
                YORK_LOG_ERROR(logger, "Exception in worker", log::exception=e);
            } catch (const std::exception& e) {
                YORK_LOG_ERROR(logger, "Exception in worker", log::exception=e);
            }
        }
    });
}

} // namespace york

