#include <internal/module.h>
#include <internal/logger/logger.h>
#include <internal/macs.h>
#include <internal/server/context.h>
#include <internal/server/handlers/create_list.h>
#include <internal/server/handlers/subscribe.h>
#include <internal/server/handlers/unsubscribe.h>
#include <internal/unsubscribe_worker/worker.h>
#include <internal/server/handlers/set_archivation_rule.h>
#include <internal/server/handlers/remove_archivation_rule.h>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>

#include <boost/algorithm/string/split.hpp>
#include <stdexcept>
#include <boost/fusion/adapted/struct/define_struct.hpp>

BOOST_FUSION_DEFINE_STRUCT((york), Pong,
        (std::string, pong) )

BOOST_FUSION_DEFINE_STRUCT((york), Error,
        (std::string, error) )

namespace york {

template <typename Handler>
auto allowOnly(Handler h, ymod_webserver::methods::http_method m) {
    return [h = std::move(h), m](std::shared_ptr<server::Context> ctx, auto logger) mutable {
        if (ctx->method() != m) {
            ctx->response().wrongMethod(Error{"method not allowed"});
            return;
        }
        h(ctx, std::move(logger));
    };
}

template <typename Handler>
auto post(Handler h) {
    return allowOnly(std::move(h), ymod_webserver::methods::mth_post);
}

template <typename Handler>
auto get(Handler h) {
    return allowOnly(std::move(h), ymod_webserver::methods::mth_get);
}

template <typename Handler>
auto bindHandler(boost::shared_ptr<ymod_webserver::server> server,
                 TVMGuardPtr tvmGuard,
                 std::vector<std::string> paths,
                 Handler handler) {
    server->bind("", paths, [tvmGuard, handler = std::move(handler)] (ymod_webserver::response_ptr stream) mutable {
        auto ctx = std::make_shared<server::Context>(stream);
        auto log = log::makeContextLog(ctx->uniqId(), ctx->requestId());
        try {
            const auto tvm = tvmGuard->check(*ctx);

            auto tvmLog = log::makeContextLog(ctx->uniqId(), ctx->requestId(), "tvm_guard");
            YORK_LOG_NOTICE(tvmLog, "tvm guard", log::endpoint=ctx->endpoint(),
                            log::tvm_guard::response=tvm);

            if (tvm.action == tvm_guard::Action::reject) {
                std::ostringstream out;
                out << tvm.reason;
                ctx->response().unauthorized(Error{out.str()});
                return;
            }
            handler(ctx, log);
        } catch (const boost::system::system_error& e) {
            const std::string msg = e.code().category().name() + std::string(" ") + std::string(e.what());
            ctx->response().internalError(Error{msg});
            YORK_LOG_ERROR_WHERE(log, "bind handler exception",
                                 log::endpoint=ctx->endpoint(),
                                 log::exception=e);
        } catch (const std::exception& e) {
            const std::string msg = std::string("exception: ") + e.what();
            ctx->response().internalError(Error{msg});
            YORK_LOG_ERROR_WHERE(log, "bind handler exception",
                                 log::endpoint=ctx->endpoint(),
                                 log::exception=e);
        } catch (...) {
            const std::string msg("unknown error");
            ctx->response().internalError(Error{msg});
            YORK_LOG_ERROR_WHERE(log, "bind handler unknown exception",
                                 log::endpoint=ctx->endpoint());
        }
    });
}

void Module::init(const yplatform::ptree& cfg) {
    auto logger = log::makeLog();
    auto tvm2Module = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("ymod_tvm");
    config = std::make_shared<Config>(cfg, tvm2Module);

    superPool.init(config->pg.pool);
    bindHandlers(cfg);

    YORK_LOG_NOTICE(logger, "module initialized");
    YORK_LOG_NOTICE(logger, "tvm guard initialized", log::tvm_guard::module=*config->tvmGuard);
}

void Module::bindHandlers(const boost::property_tree::ptree& cfg) {
    auto server = yplatform::find<ymod_webserver::server>(cfg.get<std::string>("http_server_name"));
    bindHandler(server, config->tvmGuard, {"/ping"}, [](std::shared_ptr<server::Context> ctx, auto) {
            ctx->response().ok(Pong("pong"));
        });
    bindHandler(server, config->tvmGuard, {"/create_list"}, get(
            server::handlers::makeCreateListHandler(config, getMacs())));
    bindHandler(server, config->tvmGuard, {"/subscribe"}, get(
            server::handlers::makeSubscribeHandler(config, getMacs())));
    bindHandler(server, config->tvmGuard, {"/unsubscribe"}, get(
            server::handlers::makeUnsubscribeHandler(config, getMacs())));
    bindHandler(server, config->tvmGuard, {"/set_archivation_rule"}, get(
            server::handlers::makeSetArchivationRuleHandler(config, getMacs())));
    bindHandler(server, config->tvmGuard, {"/remove_archivation_rule"}, get(
            server::handlers::makeRemoveArchivationRuleHandler(config, getMacs())));
}

void Module::start() {
    auto getSharpei = [](const SharpeiCfg& cfg, auto log) {
        return sharpei::getSharpeiClient(cfg, std::move(log));
    };
    auto io = yplatform::global_net_reactor->io();
    runUnsubscribeWorker(*io, config, getSharpei, getMacs(), getShard(), running);
}

void Module::stop() {
    running.reset();
}

}

DEFINE_SERVICE_OBJECT(york::Module)
